/*! \file uart.c

	\brief	Routines for serial Communication
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <stdio.h>
#include <stdarg.h>
#include <avr/wdt.h>
#include "display.h"
#include "uart.h"
#include "config.h"

static volatile char	tbuf1[UART1_BUFSIZE];	///< Transmit Buffer for UART1
volatile char	rbuf0[UART0_BUFSIZE];	///< Receive Buffer for UART0
static volatile char	rbuf1[UART1_BUFSIZE];	///< Receive Buffer for UART1
static volatile uint8_t	tbuf0_tail,tbuf0_head;
volatile uint8_t	tbuf1_tail,tbuf1_head;
static volatile uint8_t	rbuf0_ptr;
static volatile uint8_t	rbuf1_ptr;
static volatile uint8_t	rbuf0_cksum;
volatile char cksum0;	///< Keeps track of the Checksum for USART0. It is automatically reset by command bytes.
uint8_t charsInRbuf0;
uint8_t computedChecksum;
volatile uint8_t	byteFromHost;	///< Contains the last byte received from the host
volatile uint8_t	hostCommandsLost;	///< Counts the number of commands lost due to buffer overrun


FILE uart1=FDEV_SETUP_STREAM(putchar_uart1,NULL,_FDEV_SETUP_WRITE);


//
/*!	\brief	Initialize USART Hardware and Pointers

2 Stop-bits are used,
USART0 connects to the sensor, uses 9-bit mode and parity,
USART1 connects to the Host at 9600 baud, 8 bits and no parity.
*/
void	InitUart()
{
/*	Initialized to 0 by standard
tbuf0_tail=0;
tbuf0_head=0;
tbuf1_tail=0;
tbuf1_head=0;
rbuf0_ptr=0;
charsInRbuf0=0;
rbuf1_ptr=0;
rbuf0_cksum=0;*/
UBRR0H=(uint8_t)(SENSOR_UBRR>>8);
UBRR0L=(uint8_t)(SENSOR_UBRR);
UBRR1H=(uint8_t)(HOST_UBRR>>8);
UBRR1L=(uint8_t)(HOST_UBRR);
UCSR0B=0b10011100;	// 9-bit for UART 0
UCSR1B=0b10011000;	// RXCIE TXCIE UDRIE RXEN TXEN UCSZ2 RXB8 TXB8
UCSR0C=0b10101110;
UCSR1C=0b10000110;	// URSEL UMSEL UPM1 UPM0 USBS UCSZ1 UCSZ0 UCPOL
AssertRTS();
}

/*!	\brief	Sends a Character to USART0

USART0 does not have a separate transmit buffer and does not use interrupts.
	\param	c The Character
	\note	this function blocks if transmit buffer register is full.
*/
void put0(char c)
{
cksum0^=c;	// track checksum
Uart0EnableLineDriver();
while(!(UCSR0A & (1<<UDRE0)));	// Wait until we can put the character to the transmitter register
UDR0=c;
UCSR0B|=(1<<TXCIE0);
}

//
/*!	\brief	Sends a Command Char (bit9=1) over USART0
	\param	c The Character
*/
void putc0(char c)
{
cksum0=0;	// Reset the checksum if we transmit a command byte
UCSR0B|=(1<<TXB80);	// Set bit 9
put0(c);	// And put the character
}

// sends a data char (bit9=0) over uart0
/*!	\brief	Sends a Data Char (bit9=0) over USART0
	\param	c The Character
*/
void putd0(char c)
{
UCSR0B&=~(1<<TXB80);	// Assure that bit 9 is zero
put0(c);	// And put the character
}

/*!	\brief	USART0 Transmit Complete Interrupt

Is only used to disable the RS485 line drivers.
*/
ISR(USART0_TX_vect)
{
	Uart0DisableLineDriver();	// Transmission is complete. Disable the line driver.
}

/*!	\brief	Puts a char to USART1 (Host-Interface)

	\note	This routine blocks when the transmit buffer is full.
	\param	c The Character
	\return	 Always 0
*/
int putc1(char c)
{
uint8_t tmp;

tmp=tbuf1_head+1;
if (tmp==UART1_BUFSIZE)
tmp=0;
if ((tmp==tbuf1_tail) && !(SREG&0x80))	// buffer is full and interrupts disabled
{
	return 0;	// discard the character!
}
while (tmp==tbuf1_tail)	// buffer is full. wait...
{
	wdt_reset();
}
cli();
tbuf1[tbuf1_head]=c;
tbuf1_head=tmp;
sei();
UCSR1B|=0b01100000;			// enable UDRE and TX Complete interrupt
return 0;
}

/*!	\brief	The Putchar-Routine for USART1 (Host-Interface)

	\note	This routine blocks when the transmit buffer is full.
	\param	c The Character
	\param	stream A Pointer to the File (unused)
	\return	 Always 0
*/
int putchar_uart1(char c, FILE *stream)
{
	return putc1(c);
}

/*!	\brief	USART1 (Host-Interface) Data Register empty Interrupt
*/
ISR(USART1_UDRE_vect)
{
	Uart1EnableLineDriver();
#ifdef USE_RTS
	if (config.useRts)
	{
		if (UART1_PIN & UART1_CTS_MASK)
		{
			UCSR1B &= ~0b01100000;	// no more interrupts (UDRE and TX to keep the line driver enabled)
			return;	// Interrupts will be re-enabled by timer callback when CTS goes high
		}
	}
#endif
	UDR1=tbuf1[tbuf1_tail];
	tbuf1_tail++;
	if (tbuf1_tail==UART1_BUFSIZE)
		tbuf1_tail=0;
	if (tbuf1_head==tbuf1_tail)	// last character in buffer
		UCSR1B &= ~0b00100000;	// no more UDRE interrupts
}

/*!	\brief Tell if Transmit Buffer 1 is empty
	\return	True if empty, False if not
*/
bool	Tbub1IsEmpty(void)
{
	return (tbuf1_head==tbuf1_tail);
}

/*!	\brief	USART1 Transmit Complete Interrupt

	Transmission has been completed so the line
	driver is switched off.
*/
ISR(USART1_TX_vect)
{
	Uart1DisableLineDriver();
}

//
/*!	\brief	Receive Interrupt for USART0 (sensor)

USART0 receives not in buffered mode like
the host communication does.
USART0 resets the buffer pointer upon reception
of a 9-bit char other than 1+0x00 and directly
evaluates the buffer content on reception of
the 9-bit char 1+0x00
*/
ISR(USART0_RX_vect)
{
uint8_t bit9,c,sra;

sra=UCSR0A;
bit9=UCSR0B & 0x02;
c=UDR0;
//putchar_uart1(bit9,0);
//putchar_uart1(c,0);
if (sra&UPE0)
	cFlags|=CFLAG_PARITYERROR;
if (bit9) {	//ID-char
	if (c==0) {	// end of message, see what we've got...
		charsInRbuf0=rbuf0_ptr;
		computedChecksum=rbuf0_cksum;
		if (rbuf0_cksum!=0) {
			cFlags|=CFLAG_CHECKSUMERROR;
			return;
			}
		switch (rbuf0[0]) {
			case 'T':
				deadTime=rbuf0[4];
				pulseTimeMult=rbuf0[3];	// 1 or 8 clocks
				if ((pulseTimeMult==1) || (pulseTimeMult==8)) {	// valid multiplier?
					pulseTimeInt=(rbuf0[1]<<8)|rbuf0[2];		// yes
					cFlags|=CFLAG_TIMERECEIVED;
					}
				break;
			case 't':
				displayValues.tempSemi=rbuf0[1];
				tempRaw=rbuf0[2];
				cFlags|=CFLAG_TEMPRECEIVED;
				break;
			case '*':
				blFlags|=BOOTLDHELLO;
				break;
			case '=':
				for(uint8_t i=0;i<32;i++)
					bootLdPage[i]=rbuf0[i+1];
				blFlags|=BOOTLDDATA;
				break;
			case '-':
				blFlags|=BOOTLDERROR;
				break;
			case '+':
				blFlags|=BOOTLDOK;
				break;
			case 'P':
				blFlags|=BOOTLDPROTECTED;
				break;
			}
		rbuf0_ptr=0;
		}
	else	{
		rbuf0_ptr=0;	// a new message starts here
		rbuf0_cksum=0;	// init checksum
		}
	}
else {	// data
	}
rbuf0[rbuf0_ptr]=c;
rbuf0_cksum=rbuf0_cksum^c;
rbuf0_ptr++;
if (rbuf0_ptr>=UART0_BUFSIZE)	// overrun
	rbuf0_ptr=0;
}

void	CopyRbuf1ToCmdline(void)
{
	uint8_t	tmp;
	
	for (tmp=0;tmp<rbuf1_ptr;tmp++) {
		cmdline[tmp]=rbuf1[tmp];
	}
	cmdline[tmp]=0;	// terminating zero
}

/*!	\brief	USART1 Receive-Interrupt (Host-Interface)

This routine collects incoming characters until a complete
host- or Bluetooth-command has been received.
	\note	For Bluetooth, we ignore the high-byte of the length field. We cannot handle
			Telegrams bigger than the cmdline for now and the do also not appear until now.
*/
ISR(USART1_RX_vect)
{
uint8_t tmp,c;
static uint8_t	btLen=sizeof(rbuf1);	// Length of a bluetooth telegram

tmp=rbuf1_ptr+1;
c=UDR1;	// Get character and reset interrupt flag
byteFromHost=c;
cFlags|=CFLAG_BYTERECEIVED;
if ((rbuf1_ptr==0) && (c==10))	// The first character is a linefeed. It is an empty line!
{
	goto excmd;
}
if ((rbuf1_ptr>0) && (rbuf1[0]!=2))	// No STX, is a Host-Command
{
	if (c==13)
		return;	// ignore CRs
	if (c==10)
	{	// LF received. transfer RX-buffer to line buffer and set flag
excmd:
		if (cFlags&CFLAG_CMDRECEIVED)	// Previous command has not been processed yet
		{
			if (hostCommandsLost<255)
			{
				hostCommandsLost++;
			}
		} else	// Copy the received line to the command line
		{
			rbuf1[rbuf1_ptr]=0;	// Terminate string
			CopyRbuf1ToCmdline();
			cFlags|=CFLAG_CMDRECEIVED;	// Let main loop know
		}		
		rbuf1_ptr=0;
		rbuf1[0]=0;
		if (config.useRts)
		{
			ReleaseRTS();	// Keep the host from sending
		}
		return;
	}
} else
{	// The Bluetooth-Module is talking to us
	if (rbuf1_ptr==3)	//	This is the position of the length of the telegram
	{
		btLen=c+6;	// Here the terminating ETX will go
		if (btLen>UART1_BUFSIZE)
		{
			btLen=UART1_BUFSIZE;	// The telegram is too big and reception will fail!
		}
	}
	if (rbuf1_ptr==btLen)	// c should be an ETX here, Received a telegram from the Bluetooth-Module
	{
		CopyRbuf1ToCmdline();
		cFlags|=CFLAG_BTRECEIVED;
		tmp=0;	// So rbuf1_ptr is set to zero later
		btLen=sizeof(rbuf1);
	}
}
if (tmp==UART1_BUFSIZE)	// Buffer wraps on overflow
	tmp=0;
rbuf1[rbuf1_ptr]=c;
rbuf1_ptr=tmp;
}

void usart1FlushRx(void)
{
	rbuf1_ptr=0;
	cFlags&= CFLAG_BYTERECEIVED|CFLAG_CMDRECEIVED|CFLAG_BTRECEIVED;
}

/*!	\brief	printf_P for Communication with the Host
	\param	fmt Format String
	\param	...	Additional Parameters
	\return	The return value of the underlying vfprintf_P
*/
int hprintf_P(const char *fmt, ... )
{
	int	i;
	va_list	ap;

	va_start(ap,fmt);
	i=vfprintf_P(&uart1,fmt,ap);
	va_end(ap);
	return i;
}