﻿/*! \file sound.c
	\brief Sound and music functions

	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
	
	Playing sounds and music is done using
	timer interrupts. So the functions are non-blocking.
*/ 

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <stdint.h>
#include "sound.h"

uint8_t	sndToneCnt;	///< Down-counter for tone duration. Can be tested to 0 to wait for end of tone
uint8_t	sndSeqPlaying;	///< Is !=0 while a sequence is playing
uint8_t	*musicPtr;	///< Is used in the timer callback to play sequences

static const uint8_t midiOffset=57;	///< The first note in notes[] is midi-code 57 
const static uint16_t notes[] PROGMEM ={ ///< Contains the OCRxA-values for A3 (220Hz) to Ab8 (13310Hz)
	8378, 7908, 7464, 7045, 6650, 6277, 5924, 5592, 5278, 4982, 4702, 4438,
	4189, 3954, 3732, 3523, 3325, 3138, 2962, 2796, 2639, 2491, 2351, 2219,
	2095, 1977, 1866, 1761, 1662, 1569, 1481, 1398, 1319, 1245, 1176, 1110,
	1047, 988, 933, 881, 831, 785, 741, 699, 660, 623, 588, 555,
	524, 494, 467, 440, 416, 392, 370, 349, 330, 311, 294, 277,
	262, 247, 233, 220, 208, 196, 185, 175, 165, 156, 147, 139
	};

/*! \brief Starts playing a tone
	\param tone The value of the timer period register (OCR3A)
	
	The frequency of the tone is 3.6864MHz/(tone+1)
*/
inline void	SndToneOn(uint16_t tone)
{
	OCR3A=tone;
	TCCR3B|=1;	// direct system clock
}

/*! \brief Stops playing a tone
*/
inline void	SndToneOff()
{
	TCCR3B&=0b11111000;	// clock off
}

/*! \brief Plays a single tone
	\param tone The value of the timer period register (OCR3A)
	\param duration Length in multiple of 10ms
	
	The frequency of the tone is 3.6864MHz/(tone+1)
	A tone value 0 plays silence for the given duration
*/
void	PlayTone(uint16_t tone,uint8_t duration)
{
	if (tone!=0)
	{
		SndToneOn(tone);
	}
	else
	{
		SndToneOff();
	}
sndToneCnt=duration;
}

/*! \brief Plays a single note
	\param note The note as a midi value (e.g. A4 (440Hz) = 69)
	\param duration Length in multiple of 10ms
	
	The note value "0" can be used to play silence.
	Notes from 57 to 117 can be used. Other values
	produce undefined tones.
*/
void	PlayNote(uint8_t note,uint8_t duration)
{
	if (note!=0)
	{
		PlayTone(pgm_read_word(&notes[note-midiOffset]),duration);
	} 
	else
	{
		PlayTone(0,duration);
	}
}

/*! \brief Plays a sequence of notes
	\param seq a pointer to the note data
	
	The data format is one byte note and one
	byte duration. The data has to be terminated
	with 0x00 0x00
*/
void	PlaySequence(uint8_t *seq)
{
	PlayNote(*seq,*(seq+1));	// Tone and duration
	musicPtr=seq+2;	// Next note...
}

// Duration of notes
#define sixteenth 8
#define eigth (sixteenth*2)
#define quarter (eigth*2)
#define half (quarter*2)
#define whole (half*2)

/*!	\brief Tone sequence for error messages
*/
uint8_t	sndErrorSeq[]={57,20,0,0};

/*!	\brief Tone sequence for success
*/
uint8_t sndSuccessSeq[]={80,5,0,10,92,5,0,0};

/*!	\brief	Big Ben Tone sequence
*/
uint8_t sndBigBenSeq[]=
{
	64,quarter,68,quarter,66,quarter,59,half+quarter,0,half,
	64,quarter,66,quarter,68,quarter,64,half+quarter,0,half,
	68,quarter,64,quarter,66,quarter,59,half+quarter,0,half,
	59,quarter,66,quarter,68,quarter,64,half+quarter,0,half,
	0,0
};

/*!	\brief	Sequence for alert output 1
*/
uint8_t	sndSig1[]=
{
	76,sixteenth,81,sixteenth,76,sixteenth,81,sixteenth,76,half,0,0
};

/*!	\brief	Sequence for alert output 2
*/
uint8_t	sndSig2[]=
{
	81,sixteenth,76,sixteenth,0,0
};

