/*! \file sensors.c

	\brief	Routines for handling Sensor values
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 


#include <inttypes.h>
#include <math.h>
#include "adc.h"
#include "config.h"

/*!	\brief	Converts a P3-Raw ADC Value to Voltage
	\note	The JTAGEN-Fuse has to be cleared to receive accurate
		values for P3 since the ADC4 (PORTF4) pin will source current
		while JTAG is enabled! The error may well exceed 15% FSR.
		This has to be done before the final calibration!<br>
		During debugging via JTAG, P3 is not
		measurable and Jumper X11 has to be disconnected for the
		voltage dividers not to interfere with the JTAG interface!
	\param	val The ADC Value
	\return	The Voltage
*/
double	P3FromADC(uint16_t val)
{
	return val*config.P3VoltsPerLSB;
}

/*!	\brief	Converts a P5-Raw ADC Value to Voltage
	\param	val The ADC Value
	\return	The Voltage
*/
double	P5FromADC(uint16_t val)
{
	return val*config.P5VoltsPerLSB;
}

/*!	\brief	Converts a Unreg-Raw ADC Value to Voltage
	\param	val The ADC Value
	\return	The Voltage
*/
double	UnregFromADC(uint16_t val)
{
	return val*config.UnregVoltsPerLSB;
}

/*!	\brief	Converts a SWUnreg-Raw ADC Value to Voltage
	\param	val The ADC Value
	\return	The Voltage
*/
double	SWUnregFromADC(uint16_t val)
{
	return val*config.SWUnregVoltsPerLSB;
}

/*!	\brief	Converts the ADC-Temp Value to Celsius

	\note	This function takes extensive floating point operations
			and may thus block for about 400us.
	\return	 The Temperature in Celsius
*/
double	TempFromADC(void)
{
	const double rVcc=47000.0;
	const double r25=10000.0;
	const double a1=0.003354016;
	const double b1=0.0002828750;
	const double c1=2.989645E-06;
	const double d1=4.840855E-08;
	const double celsius0=273.15;

	double	untc=ADCData.named.ADC_Temp*(2.56/1024);
	double	r=rVcc/(P5FromADC(ADCDataAvg.named.ADC_P5)-untc)*untc;
	double	lnr=log(r/r25);
	double	lnr2=lnr*lnr;
	return 1/(a1+b1*lnr+c1*lnr2+d1*lnr2*lnr)-celsius0;
}

