/*
 * Key.h
 *
 * Created: 25.11.2012 20:54:52
 *  Author: Robert
 */ 

#include <stdint.h>
#include <stdbool.h>
#include <avr/io.h>

#ifndef KEY_H_
#define KEY_H_

extern volatile uint8_t kflags;
static const uint8_t KFLAG_KEYPRESSED=1;	///< The key has been pressed
static const uint8_t KFLAG_KEYRELEASED=2;	///< The key has been released
static const uint8_t KFLAG_KEYPRESSEDSHORT=4;	///< The key has been pressed and released within 2 seconds
static const uint8_t KFLAG_KEYPRESSEDLONG=8;	///< The key has been pressed for more than 2 seconds

/*!	\brief	Returns the current state of the Pushbutton Key
	\return	True if the Key is pressed, else false
*/
static inline bool	GetKeyState(void)
{
	return !(PINB&0x01);
}


/*!	\brief	Must be called from the 100Hz-Timer Interrupt

	kflags are set in the following order:<br>
	KEY_PRESSED	as soon as a keypress is detected<br>
	KEY_PRESSEDSHORT	the key has been released but has not been pressed for at least 2.5 seconds<br>
	KEY_PRESSEDLONG	the Key is pressed for 2.5 seconds or more (but not necessarily released yet).<br>
	KEY_RELEASED when the key is released<br>
	Only the KEY_PRESSED and KEY_RELEASED events are immediate.
	The others are set after the key has been released
	or after 2.5 seconds.<br>
	The times have a granularity of 100ms (i.e. KEY_PRESSED may be detected 99ms after the key
	has really been pressed and the 2.5 seconds may have been 2.50 or 2.599 seconds) but that's absolutely o.k. for a UI.
*/
static inline void	KeyInt(void)	///< Must be called from 10ms-interrupt
{
	static bool	lastKeyState=false;	// Key state at last interrupt
	static	uint8_t	keyPressedTime=0;	// Counts the time the key is pressed
	static	uint8_t	cnt100ms=0;	// Counts the 100ms-intervals
	bool	keyState;	// current key state
	
	cnt100ms++;
	if (cnt100ms>=10)
	{
		cnt100ms=0;
	} else
	{
		return;
	}
	// The following is executed every 100ms!
	keyState=GetKeyState();
	if (lastKeyState!=keyState)	// Something has changed
	{
		lastKeyState=keyState;	// Keep the key state for next interrupt
		if (keyState)	// Key has just been pressed
		{
			kflags|=KFLAG_KEYPRESSED;
			keyPressedTime=0;
			return;
		} else	// Key has just been released
		{
			kflags|=KFLAG_KEYRELEASED;
			if (keyPressedTime<25)
			{
				kflags|=KFLAG_KEYPRESSEDSHORT;
			}
		}
	}
	if (keyState) {
		if (keyPressedTime<255)
		{
			keyPressedTime++;
		}
		if (keyPressedTime==25)	// More than 2.5 seconds
		{
			kflags|=KFLAG_KEYPRESSEDLONG;
		}
	}	
}

#endif /* KEY_H_ */