// Display.h

#ifndef DISPLAY_H_
#define DISPLAY_H_

#include <stdbool.h>
#include <avr/io.h>
#include "lcd.h"

#define WITH_UPDATE		///< Compile with Sensor Update Function

#define SIGNAL_PORT	PORTA
#define SIGNAL_1_MASK	0b00000100
#define SIGNAL_2_MASK	0b00001000

typedef double	LITERS_T;	///< Remember that liters may go negative

LITERS_T	LitersFromDist(double dist);
void		ResetDisplayFlags(void);

/*!	\brief  Holds all values used for Display Purposes

	Display routines should only use values of this struct and every value
	you want to be displayed should be placed here. By this mechanism we can
	decouple updating values from displaying values.<br>
	Exceptions are e.g. the ADC-values for the display sensors as it does not
	make sense to duplicate them here.<br>
	There are some char-values for error flags. They contain the character value
	to be displayed on the corresponding position (i.e. SPACE if no error persists).
	They are set asynchronously but if the current page needs them, it takes
	them from here (just like all the other values).
*/
typedef struct  
{
	LITERS_T	liters;	///< The current volume in liters
	uint8_t		literUnitChar;	///< Holds a small or capital letter 'L'
	LITERS_T	litersPerHour;	///< The volume change for the running hour
	LITERS_T	litersLastHour;	///< The volume for the last hour
	LITERS_T	litersToday;	///< The volume change for today
	LITERS_T	litersYesterday;	///< The volume at last midnight
	double		litersPerCent;	///< Filling of the cistern in percent
	int8_t		tempSemi;		///< The temperature in 0.5C-units
	LITERS_T	litersTemp;	///< For averaging
	uint8_t		nlitersTemp;	///< Number of values added to liters_temp
	char		parityErrorChar;	///< Character to appear on Position "Parity Error"
	char		checksumErrorChar;	///< Character to appear on Position "Checksum Error"
	char		sensorDeadChar;	///< Character to appear on Position "SensorDead"
	char		not10ErrorChar;	///< Character to appear on Position "More or less than 10 Values received"
	char		*error;	///< Pointer to a string describing the most urgent error
} displayValues_t;

extern displayValues_t displayValues;
extern uint16_t		pulseTimeInt;
extern uint8_t		pulseTimeMult;
extern uint8_t		deadTime;
extern signed char	tempSemi;
extern uint8_t		tempRaw;
extern volatile char	cmdline[81];
extern double	lastReportedVolume;

extern volatile uint8_t	tickCnt,secCnt,minCnt,hrCnt;

extern uint8_t	wdrCnt;
extern uint8_t	boCnt;

/// Current speed of the RTC (Real Time Clock)
typedef enum
{
	RTC_NORMAL,	///< RTC is running at normal speed (realtime)
	RTC_FASTER,	///< RTC is running faster by 1%
	RTC_SLOWER	///< RTC is running slower by 1%
} RTCAdj_t;
extern const char	*RTCAdjTxt[];
extern volatile RTCAdj_t	RTCAdj;

/// Command for ShowScreen()
typedef enum
{ 
	SCREENSTEPAUTO,	///< Advance to the next page of the frequently used
	SCREENSTEPMANU, ///< Advance to the next of all available pages
	SCREENREFRESH,	///< Do not change the page, just refresh the current one
	SCREENABSPAGE,	///< Show a special page. Currently only used for the init-screen.
} showScreen_t;

#define	SCREENABSPAGE_VERSION	128	///< Display Version Screen at startup

#define ADCVoltsPerLSB (2.56/1024)	///< Voltage step for one LSB of the AD converter
#define	dividerP3 (4.7/(4.7+8.2))	///< Resistor divider for P3. Gives 7.03V FSR or 6.86mV/LSB
#define dividerP5 (4.7/(4.7+8.2))	///< Resistor divider for P5. Gives 7.03V FSR or 6.86mV/LSB
#define dividerUnreg (4.7/(4.7+68))	///< Resistor divider for Unreg. Gives 39.6V FSR or 38.7mV/LSB
#define	dividerSWUnreg (4.7/(4.7+68))	///< Resistor divider for SWUnreg. Gives 39.6V FSR or 38.7mV/LSB

extern uint8_t	bootLdPage[];
extern uint16_t		bootLdAdr;
extern volatile uint8_t	tFlags;
static const uint8_t TFLAG_SEC=1;	///< Is set once per Second
static const uint8_t TFLAG_MIN=2;	///< Is set once per Minute at Second 0
static const uint8_t TFLAG_HOUR=4;	///< Is set once per Hour at Minute 0
static const uint8_t TFLAG_DAY=8;	///< Is set every Day at Midnight


extern volatile uint8_t cFlags;
static const uint8_t CFLAG_TIMERECEIVED=1;	///< Received a Travel Time from the Sensor
static const uint8_t CFLAG_TEMPRECEIVED=2;	///< Received the current Sensor Temperature
static const uint8_t CFLAG_PARITYERROR=4;	///< Parity Error in received Byte
static const uint8_t CFLAG_CHECKSUMERROR=8;	///< Checksum Error in received Telegram
static const uint8_t CFLAG_CMDRECEIVED=16;	///< Received the terminating CR of a Command
static const uint8_t CFLAG_BYTERECEIVED=32;	///< Received a Byte from the host
static const uint8_t CFLAG_BTRECEIVED=64;	///< Received a telegram from the Bluetooth-Module

extern volatile uint8_t blFlags;
static const uint8_t BOOTLDHELLO=1;	///< Boot Loader Hello received ("*")
static const uint8_t BOOTLDDATA=2;	///< Boot Loader Data received ("=")
static const uint8_t BOOTLDOK=4;	///< Boot Loader Data received ("+")
static const uint8_t BOOTLDERROR=8;	///< Boot Loader Error received ("-")
static const uint8_t BOOTLDPROTECTED=16;	///< Boot Loader Protected Area received ("P")

extern volatile uint8_t	dFlags;
static const uint8_t DFLAG_DCF77=1;		///< Gives information about DCF77 and RTC
static const uint8_t DFLAG_SENSOR=2;	///< Gives information about Sensor Communication
static const uint8_t DFLAG_ADC=4;	///< Gives information about ADC MIN/MAX Values

extern double	pulseTime;
extern double	distance;
extern double	speedOfSound;

/// Constants for Display Type
typedef enum
{
	DISPLAY_2x16,	///< Display is a Winstar WH1602B3 or equivalent
	DISPLAY_4x20,	///< Display is a Winstar WH2004A or equivalent
	DISPLAY_ANALOG,	///< We have no LCD at all, only analog output
} displayType_t;
extern displayType_t	displayType;
/*!	\brief Parameters for each of the possible Displays
*/
typedef struct {
	uint8_t	width;	///< The width of the display in characters
	uint8_t	nLines;	///< The number of lines
	uint8_t	startLine[LCD_MAX_LINES];	///< Offset of character 0 of each line
}	displayMetrics_T;
extern const displayMetrics_T displayMetrics[];

/*!	\brief	Switches the Sensor Power on
*/
inline static void SensorPowerOn(void)
{
	PORTA |= 0x01;
}

/*!	\brief	Switches the Sensor Power off
*/
inline static void SensorPowerOff(void)
{
	PORTA &= ~0x01;
}

/*!	\brief	Switches the Bluetooth Power on
*/
inline static void BluetoothPowerOn(void)
{
	PORTD |= 0xC0;
}

/*!	\brief	Switches the Bluetooth Power off
*/
inline static void BluetoothPowerOff(void)
{
	PORTD &= ~0x80;
}

/*!	\brief	Switches Signal Output Optocoupler 1 to ON
*/
inline static void	Sig1ON (void)
{
	SIGNAL_PORT&= ~SIGNAL_1_MASK;
}

/*!	\brief	Switches Signal Output Optocoupler 1 to OFF
*/
inline static void Sig1OFF (void)
{
	SIGNAL_PORT|= SIGNAL_1_MASK;
}

/*!	\brief	Switches Signal Output Optocoupler 2 to ON
*/
inline static void	Sig2ON (void)
{
	SIGNAL_PORT&= ~SIGNAL_2_MASK;
}

/*!	\brief	Switches Signal Output Optocoupler 2 to OFF
*/
inline static void Sig2OFF (void)
{
	SIGNAL_PORT|= SIGNAL_2_MASK;
}

/*!	\brief	Gets State of Optocoupler 1
	True means Optocoupler is ON
*/
inline static bool GetSig1(void)
{
	return !(SIGNAL_PORT&SIGNAL_1_MASK);
}

/*!	\brief	Gets State of Optocoupler 2
	True means Optocoupler is ON
*/
inline static bool GetSig2(void)
{
	return !(SIGNAL_PORT&SIGNAL_2_MASK);
}

typedef	uint8_t	delay_t;

extern void	delay10ms(delay_t delay10ms);

extern void	SetAnalogPerCent(double perCent);

#endif