/*! \file display.c

	\brief	The Main Program and miscellaneous Routines
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <avr/interrupt.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <avr/pgmspace.h>
#include "adc.h"
#include "adeunis.h"
#include "config.h"
#include "delay.h"
#include "display.h"
#include "ihex.h"
#include "lcd.h"
#include "uart.h"
#include "dcf77.h"
#include "usartcmd.h"
#include "key.h"
#include "led.h"
#include "sound.h"
#include "ftest.h"
#include "sensors.h"
#include "stackchk.h"

/*!	\brief	Fuse settings for the Production ELF file
*/
FUSES =
{
	.low = (FUSE_BODLEVEL & FUSE_BODEN),
	.high = (FUSE_SPIEN & FUSE_BOOTSZ0 & FUSE_BOOTSZ1),
	.extended = (0xff),
};

LOCKBITS = (LB_MODE_1);	///< The Lock Bits for the production File

const char* build= __DATE__;	///< The Build Date as to be displayed in the Boot Screen
volatile uint8_t	tickCnt;	///< counter for timer0 interrupts
volatile uint8_t	secCnt;		///< Seconds of the RTC
volatile uint8_t	minCnt;		///< Minutes of the RTC
volatile uint8_t	hrCnt;		///< Hours of the RTC
volatile RTCAdj_t	RTCAdj;		///< Soft Adjustment of the RTC (+-1%)
const char	*RTCAdjTxt[]={"normal","faster","slower"};	///< Human readable interpretation of RTCAdj

/*!	\brief	Flags set by Timer Interrupt to signal Main Loop

	The flags are set each second, minute, hour or day respectively by timer interrupt.
	The flags must be cleared by the main loop.
*/
volatile uint8_t	tFlags;

/*!	\brief	Flags set by Serial Communication with the Sensor

	The flags are set by serial interrupt and
	must be cleared by the main loop
*/
volatile uint8_t	cFlags;

volatile uint8_t	blFlags;		///< Boot Loader Communication Flags
volatile uint8_t	dFlags;			///< Flags Controlling Debug Output

bool	discardFlightTime;	///< Used to discard the first measurement after sensor power on

BTStatus_t	BTStatus;	///< Status for Bluetooth connection

//! Metrics for each of the possible displays
const displayMetrics_T	displayMetrics[2]=
	{
		{16,2,{0,64,20,84}},	// We define four lines here also, so we can print information only visible on the 4x20
		{20,4,{0,64,20,84}}
	};
// Reset Counters in section .noinit because they must survive a reset!
uint8_t	wdrCnt __attribute__ ((section (".noinit")));	///< Counts the number of Watchdog Resets
uint8_t	boCnt __attribute__ ((section (".noinit")));	///< Counts the number of Brown Out Resets

displayValues_t	displayValues;	///< Holds all values that are used for display
uint16_t	pulseTimeInt;	///< Traveling time of the last pulse in timer1-clocks of the sensor CPU
uint8_t	deadTime;	///< Received dead time value from the sensor
double	speedOfSound;	///< In m/s, temperature calibrated
double	pulseTime;	///< In seconds
uint8_t	pulseTimeMult; ///< 1 or 8 clocks from sensor timer1 prescaler
double	distance;	///< To the reflecting surface in meters
uint8_t	tempRaw;	///< Raw value from the ADC
static bool sensorDead;	///< Is set to false when a sensor telegram arrives.
double	lastReportedVolume;	///< Last Volume spontaneously reported.

volatile char	cmdline[81]; ///< Ready to hold 80 chars plus terminating 0

bool	backlightAlwaysOn;	///< Shall the Backlight be always on?
uint8_t	backlightOnTime;	///< Time in Seconds the LCD-Backlight is already on (topped at 255 Seconds)
static uint8_t displayTimer;	///< Counts the seconds for each display page

uint8_t	bootLdPage[32];	///< Holds the data for boot loader write/read
uint16_t	bootLdAdr;		///< The address of the page

static volatile delay_t	delaycnt;	///< Counter for delay()

char	displayError[80];

/*!	\brief	Resets the Error Flags in the Display

	Replaces the character type members flag characters (aka "<PC") by
	printable default values.
	\note	This function has to be called at least during system
	initialization since the characters would be 0 by default.
	This would terminate any line being displayed at the position they reside!
*/
void ResetDisplayFlags(void)
{
	displayValues.parityErrorChar=' ';
	displayValues.checksumErrorChar=' ';
	displayValues.not10ErrorChar=' ';
	displayValues.sensorDeadChar=' ';
	displayValues.literUnitChar='L';
	displayValues.error=NULL;
}

/*!	\brief	Initializes the System
*/
static void	Init(void)
{
	// First action: determine cause of reset...
	if (MCUCSR&(1<<PORF)) {	// Power on reset
		wdrCnt=0;
		boCnt=0;
		}
	else if (MCUCSR&(1<<WDRF)) {	// Watchdog-Reset
		if (wdrCnt<255)
			wdrCnt++;
		}
	else if (MCUCSR&(1<<BORF)) {	// Brownout-Reset
		if (boCnt<255)
			boCnt++;
		}
	MCUCSR=0;
	ReadConfig();
	DDRA=0b11111101;	// Only DCF77 is input
	DDRB=0b11111110;	// Only Key is input, all unused lines output
	PORTB|=0b00000001;	// Enable Pull-up on Key
	//DDRC is initialized by InitLCD
	DDRD=0b11011011;	// Only RxD and CTS are inputs
	PORTD|=0b00000100;	// enable pull-up on RxD1
	DDRE=0b11111110;	// Only RxD is input
	DDRF=0b11100000;	// Voltage dividers as input
	DDRG=0b11111111;	// unused, set to output
	wdt_enable(WDTO_2S);
	TCCR0=0b00001111;	// Timer0 CTC-Mode, Clk/1024
	OCR0=35;			// gives 100Hz
	TIMSK|=0b00000010;	// enable timer0 Output Compare Match Interrupt
	TCCR1A=0b10101101;	// Non-inverting PWM, Fast PWM 8-bit
	TCCR1B=0b00001001;	// Direct System Clock
	OCR1AL=config.LCDBright;	// set LCD brightness
	OCR1BL=config.LCDV0;		// default LCD-Voltage
	displayValues.tempSemi=(int8_t)0x80;
	speedOfSound=340.0;
	backlightOnTime=config.backlightOnTime;
	backlightAlwaysOn=config.backlightAlwaysOn;
	ResetDisplayFlags();
	InitSound();
	InitLCD();
	InitUart();
	InitADC();
	InitDCF77();
	BluetoothPowerOn();
	stdout=&lcd;
	sei();	// Enable Interrupts. We are ready now!
}

/*!	\brief	Timer0 compare interrupt (100Hz)

	The 100Hz-interrupt is not only used to create a RTC (Real Time Clock)
	but also for other tasks that rely on a constant timing, i.e. the
	DCF77 routines and debouncing of the pushbutton key.<br>
	If you change the frequency, be sure of all of the consequences!
*/
ISR(TIMER0_COMP_vect)
{
	LEDYellowOn();	// Indicate that we are processing the Timer Interrupt
	if (delaycnt)	// for delay10ms()
	{
		delaycnt--;
	}
	if (tickCnt==0 && RTCAdj==RTC_FASTER)
	{
		tickCnt++;	// Add another tick so this 'second' will only be 0.99s
	}
	tickCnt++;
	sei();	// Enable interrupts since we might take a little longer...
	if (tickCnt==100) {	// One second has passed
		tickCnt=0;
		if (RTCAdj==RTC_SLOWER)
		{
			tickCnt=0xff;	// Set tickcnt to -1 so the next 'second' will be 1.01s
		}
		tFlags|=TFLAG_SEC;
		secCnt++;
		if (secCnt==60) {	// One minute has passed
			secCnt=0;	// Reset second counter
			tFlags|=TFLAG_MIN;
			minCnt++;
			if (minCnt==60) {	// One hour has passed
				minCnt=0;	//	Reset minute counter
				tFlags|=TFLAG_HOUR;
				hrCnt++;
				if (hrCnt==24) {	// One day has passed
					hrCnt=0;	// Reset hour counter
					tFlags|=TFLAG_DAY;	// And set day-flag. we don't care for longer time periods
					}
				}
			}
		}
	DCF77int();
	KeyInt();	// Debouncing the key
	SndTimerCallback();	// Timing of music playback
	UsartTimerCallback();	// Necessary for RTS/CTS-Handshake
	LEDYellowOff();	// We're done
}

/*!	\brief	limits an integer from -9999 to +9999
			for display purposes
	\param	i the value to limit
	\return	i if in range, else -9999 or +9999
*/
static int16_t	IntLimit(LITERS_T i)
{
	if (i<-9999)
		return -9999;
	else if (i>9999)
		return 9999;
	else return i;
}

/*!	\brief	Prepares the lcdVbarChar[]-Array
	\return	The return Value of LCDBuildVBarChars (true if OK, false if not)
*/
bool	VBarGraphPrepare(void)
{
	uint8_t	bars;
	
	bars=displayValues.litersPerCent/100.0*displayMetrics[config.displayType].nLines*8.0+0.5;
	return LCDBuildVBarChars(bars);
}

/*!	\brief	Draws a horizontal Bar Graph showing the Cistern Filling in %
	\param	line	The 0-based Line Number
	\return	The return Value of LCDBarGraph (-1 on Error, 0 if OK)
*/
int8_t	HBarGraphPerCent(uint8_t line)
{
	uint8_t	bars;
	
	// litersPerCent is guaranteed to be between 0.0 and 100.0
	bars=displayValues.litersPerCent/100.0*displayMetrics[config.displayType].width*5.0+0.5;
	return LCDHBarGraph(line,bars);
}

/*!	\brief  Steps through various Screens (or Pages) to be displayed

	The static variable <i>page</i> is automatically incremented on each call
	(except of SCREENREFRESH) and automatically resets to 0 if no <i>case</i>
	exists for the value. This makes the function really handy. If you want
	to add a page, simply add a <i>case</i> with the next higher value,
	and insert the code to display it, that's all.
	\param	action	What to do:<ul>
		<li>SCREENSTEPAUTO steps through the most frequent pages</li>
		<li>SCREENSTEPMANU additionally displays some extra pages normally not needed</li>
		<li>SCREENREFRESH just redraws the current page with actual values</li>
		<li>SCREENABSPAGE show a special page according to pageNo</li></ul>
	\param	pageNo	The zero-based index of the Page to be displayed.
		Only used if action==SCREENABSPAGE
	\note Each screen should assure that no previous artifacts remain, i.e.
		it should either display four (or two, according to the display type) complete lines,
		including trailing spaces, or clear the display before drawing.<br>
		To assure a 'quiet' display without flicker redrawing all lines of the display is the preferred method.
		ShowScreen may be called several times a second.<br>
		Remaining lines or characters from previous screens would not be updated and might irritate the user.<br>
		So redrawing each line via LCDPrintLine_P() is the method of choice.
*/
static void	ShowScreen(showScreen_t	action, uint8_t pageNo)
{
	static uint8_t page;

	if (action==SCREENABSPAGE)	// Displaying a special Page No. is requested
	{
		page=pageNo;
	} else if (action!=SCREENREFRESH)	// Refresh keeps the Page No.
	{
		page++;	// All others advance to the next Page
	}
again:
	switch (config.displayType) {
		case DISPLAY_ANALOG:	// nothing to do
			break;
		case DISPLAY_2x16:
			if (page<6)	// The first line is equal for the first pages
			{
				LCDPrintLine_P(0,PSTR("%4d%c%c%c%c%2d:%02d:%02d   %c"),IntLimit(displayValues.liters),displayValues.literUnitChar,
				displayValues.not10ErrorChar,displayValues.parityErrorChar,displayValues.checksumErrorChar,
				hrCnt,minCnt,secCnt,DCFInSync?1:' ');
			}
			LCDGoTo(displayMetrics[DISPLAY_2x16].startLine[1]);	// line 2 - all additional information goes here!
			switch (page) {
				case 0:
					if ((wdrCnt==0) && (boCnt==0))
					{
						page++;	// No WDR or Brownout. Skip this page
						goto again;
					}						
					printf_P(PSTR("W%03d  B%03d      "),wdrCnt,boCnt);
					break;
				case 1:
					printf_P(PSTR("%+5d l/h %+5d*"),IntLimit(displayValues.litersPerHour),IntLimit(displayValues.liters-displayValues.litersLastHour));
					break;
				case 2:
					printf_P(PSTR("%+5d l/d %+5d*"),IntLimit(displayValues.litersToday),IntLimit(displayValues.liters-displayValues.litersYesterday));
					break;
				case 3:
					printf_P(PSTR("%5.1f\337C %5.1fm/s"),(double)displayValues.tempSemi/2.0,speedOfSound);
					break;
				case 4:
					printf_P(PSTR("%5.3fms %5.3fm  "),pulseTime*1000.0,distance);
					break;
				case 5:
					HBarGraphPerCent(1);
					break;
				// extended pages only accessible via key
				case 6:
					if (action==SCREENSTEPAUTO)
					{
						page=0;	// Sequence for automatic display ends here. Start over with page 0
						goto again;
					}					
					LCDPrintLine_P(0,PSTR("P5:      %5.3fV"),P5FromADC(ADCDataAvg.named.ADC_P5));
					LCDPrintLine_P(1,PSTR("P3:      %5.3fV"),P3FromADC(ADCDataAvg.named.ADC_P3));
					break;
				case 7:
					LCDPrintLine_P(0,PSTR("Unreg:   %5.2fV"),UnregFromADC(ADCDataAvg.named.ADC_Unreg));
					LCDPrintLine_P(1,PSTR("SWUnreg: %5.2fV"),SWUnregFromADC(ADCDataAvg.named.ADC_SWUnreg));
					break;
				case 8:
					LCDClear();
					LCDPrintLine_P(0,PSTR("Disp.Temp: %-4.1f\337C"),TempFromADC());
					break;
				// The Startup-Screen
				case SCREENABSPAGE_VERSION:
					LCDClear();
					LCDPrintLine_P(0,PSTR("Usdist 2.0"));
					LCDPrintLine_P(1,PSTR("Bld %s"),build);
					break;
				default:
					if (action!=SCREENREFRESH)	//	Refresh only, page number did not change
					{
						page=0;	// Invalid page number, switch to page 0
						goto again;
					}
				}
			break;
		case DISPLAY_4x20: // the sequence for the four-line-display is shorter...
			if (page<3)	// The first three lines are equal for the first three pages
			{
				LCDPrintLine_P(0,PSTR("%4d%c%c%c%c%2d:%02d:%02d %c%c%c"),IntLimit(displayValues.liters),displayValues.literUnitChar,
					displayValues.not10ErrorChar,displayValues.parityErrorChar,displayValues.checksumErrorChar,
					hrCnt,minCnt,secCnt,displayValues.sensorDeadChar,DCFInSync?1:' ',
					lcdVbarChar[3]);
				LCDPrintLine_P(1,PSTR("% 5d  l/h % 5d*  %c"),IntLimit(displayValues.litersPerHour),
					IntLimit(displayValues.liters-displayValues.litersLastHour),
					lcdVbarChar[2]);
				LCDPrintLine_P(2,PSTR("% 5d  l/d % 5d*  %c"),IntLimit(displayValues.litersToday),
					IntLimit(displayValues.liters-displayValues.litersYesterday),
					lcdVbarChar[1]);
			}
			switch (page) {
				case 0:
					if ((wdrCnt==0) && (boCnt==0))
					{
						page++;	// No WDR or Brownout. Skip this page
						goto again;
					}						
					LCDPrintLine_P(3,PSTR("W%03d   B%03d        %c"),wdrCnt,boCnt,lcdVbarChar[0]);
					break;
				case 1:
					LCDPrintLine_P(3,PSTR("%5.1f\337C   %5.1fm/s %c"),(double)displayValues.tempSemi/2.0,speedOfSound,lcdVbarChar[0]);
					break;
				case 2:
					LCDPrintLine_P(3,PSTR("%6.3fms %7.2fmm %c"),pulseTime*1000.0,distance*1000.0,lcdVbarChar[0]);
					break;
				// extended pages only accessible via key
				case 3:
					if (action==SCREENSTEPAUTO)
					{
						page=0;	// Sequence for automatic display ends here. Start over with page 0
						goto again;
					}
					LCDPrintLine_P(0,PSTR("P5:      %5.3fV"),P5FromADC(ADCDataAvg.named.ADC_P5));
					LCDPrintLine_P(1,PSTR("P3:      %5.3fV"),P3FromADC(ADCDataAvg.named.ADC_P3));
					LCDPrintLine_P(2,PSTR("Unreg:   %5.2fV"),UnregFromADC(ADCDataAvg.named.ADC_Unreg));
					LCDPrintLine_P(3,PSTR("SWUnreg: %5.2fV"),SWUnregFromADC(ADCDataAvg.named.ADC_SWUnreg));
					break;
				case 4:
					LCDPrintLine_P(0,PSTR("DisplayTemp: %-4.1f\337C"),TempFromADC());
					if (displayValues.error!=NULL)
					{
						LCDPrintLine_P(1,PSTR("%s"),displayValues.error);
					}
					Stackchk();
					LCDPrintLine_P(2,PSTR("Stack %04X Free:%d"),stackchk.stackmin,stackchk.stackmin-stackchk.heapmax-1);
					HBarGraphPerCent(3);
					break;
				case SCREENABSPAGE_VERSION:
					LCDClear();
					LCDPrintLine_P(0,PSTR("Usdist 2.0"));
					LCDPrintLine_P(1,PSTR("Build %s"),build);
					break;
				default:
					if (action!=SCREENREFRESH)	//	Refresh only, page number did not change
					{
						page=0;	// Invalid page number, switch to page 0
						goto again;
					}
				}
			break;
		}
}

/*!	\brief	Computes the Volume in Liters from the Distance in Meters
	\param	dist The distance in Meters
	\note	The calculation of the volume is, especially with horizontal
		cylinders, quite complex and may take more than a millisecond of
		CPU time.<br>
		If dist is zero, liters will be returned as zero to indicate
		an invalid value. This is not fail safe since it <b>might</b> be
		exactly zero but the user will notice that the liters are invalid.
	\return	The Volume in Liters
*/
LITERS_T	LitersFromDist(double dist)
{
	double	r,volume,level;
	
	if (dist<1e-3)	// Distance is zero, no valid measurement
	{
		return 0;
	}
	level=config.cistHeight-dist;
	r=config.cistAreaOrDiameter/2.0;
	switch(config.cisternType) {
		case CIST_CONST_AREA:
			volume=level*config.cistAreaOrDiameter*1000.0+0.5;
			break;
		case CIST_HORIZ_CYL:
			volume=config.cistLength*(r*r*acos(1-level/r)-
				sqrt(2*r*level-level*level)*(r-level));
			break;
		case CIST_SPHERE:
			volume=level*level*M_PI/3.0*(3*r-level);
			break;
		default:
			volume=0.0;
	}		
	return volume;
}


/*!	\brief	Checks for Communication Flags

	\note Be sure to check and clear all Flags even if you don't
	use them to avoid unnecessary calls of this function
*/
static void DoCflags(void)
{
	uint8_t	i;
	static BTPacket_t	*btPacket;
	LITERS_T	receivedLiters;
	
	if (cFlags&CFLAG_CHECKSUMERROR) {
		cFlags&=~CFLAG_CHECKSUMERROR;
		displayValues.checksumErrorChar='C';
		ShowScreen(SCREENREFRESH,0);
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("%02d.%02d0 Checksum Error\n"),secCnt,tickCnt);
			for (i=0;i<charsInRbuf0;i++)
			{
				hprintf_P(PSTR("%02X "),rbuf0[i]);
			}
			hprintf_P(PSTR(" : %02X\n"),computedChecksum);
		}
	}
	if (cFlags&CFLAG_PARITYERROR) {
		cFlags&=~CFLAG_PARITYERROR;
		displayValues.parityErrorChar='P';
		ShowScreen(SCREENREFRESH,0);
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("%02d.%02d0 Parity Error\n"),secCnt,tickCnt);
		}
	}
	if (cFlags&CFLAG_TIMERECEIVED) {	// new travel time results
		cFlags&=~CFLAG_TIMERECEIVED;
		sensorDead=false;	// It lives!
		displayValues.sensorDeadChar=' ';
		if (!discardFlightTime)
		{
			pulseTime=(double)pulseTimeMult/F_CPU*(double)pulseTimeInt;
			distance=speedOfSound*pulseTime/2.0;
			receivedLiters=LitersFromDist(distance);
			displayValues.liters=receivedLiters;
			displayValues.litersTemp+=receivedLiters;
			displayValues.nlitersTemp++;
			displayValues.literUnitChar='l';	// A small letter 'L' indicates measurement in progress
			ShowScreen(SCREENREFRESH,0);
		}	else {
			discardFlightTime=false;	// Do not discard from now
		}
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("%02d.%02d0 Time: %6.3fms %6.4fm %dx deadTime:%d liters:%5.1f\n"),
				secCnt,tickCnt,pulseTime*1000.0,distance,pulseTimeMult,deadTime,displayValues.litersTemp/displayValues.nlitersTemp);
		}		
		if (displayValues.nlitersTemp==10) {	// Enough values received for averaging
			SensorPowerOff();
			displayValues.litersTemp+=5.0;	// Rounding
			displayValues.liters=displayValues.litersTemp/(double)displayValues.nlitersTemp;
			displayValues.litersPerCent=displayValues.liters/config.cistCapacity*100.0;
			if (displayValues.litersPerCent>100.0)	//	Ensure a value between 0 and 100%
			{
				displayValues.litersPerCent=100.0;
			} else if (displayValues.litersPerCent<0.0)
			{
				displayValues.litersPerCent=0.0;
			}
			SetAnalogPerCent(displayValues.litersPerCent);	// Set the analog value
			VBarGraphPrepare();
			displayValues.literUnitChar='L';	// A capital letter 'L' indicates averages
			ShowScreen(SCREENREFRESH,0);
			if ((displayValues.liters>=(LITERS_T)config.signalOutputs[0].activeBelow) && (displayValues.liters<=(LITERS_T)config.signalOutputs[0].activeAbove))
			{
				Sig1OFF();
			} else
			{
				Sig1ON();
			}
			if ((displayValues.liters>=(LITERS_T)config.signalOutputs[1].activeBelow) && (displayValues.liters<=(LITERS_T)config.signalOutputs[1].activeAbove))
			{
				Sig2ON();
			} else
			{
				Sig2OFF();
			}
		}
	}
	if (cFlags&CFLAG_TEMPRECEIVED) {
		cFlags&=~CFLAG_TEMPRECEIVED;
		speedOfSound=331.5*sqrt((displayValues.tempSemi/2.0+273.16)/273.16);
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("%02d.%02d0 Temp: %3d %+4.1fC speedOfSound:%5.1f\n"),secCnt,tickCnt,tempRaw,(double)displayValues.tempSemi/2.0,speedOfSound);
		}
	}
	if (cFlags & CFLAG_CMDRECEIVED) { // lets look what the user wants...
		cFlags&=~CFLAG_CMDRECEIVED;
		ExecCommand((char*)cmdline);
		AssertRTS();
	}
	if (cFlags & CFLAG_BYTERECEIVED) // We are not interested in every single byte here
	{
		cFlags&=~CFLAG_BYTERECEIVED;
	}
	if (cFlags & CFLAG_BTRECEIVED) // We are not interested in Bluetooth affairs for now
	{
		cFlags&=~CFLAG_BTRECEIVED;
		btPacket=(BTPacket_t *)(&cmdline[1]);
		switch (btPacket->packetType)
		{
			case ADEUNIS_IND:
				switch (btPacket->opcode)
				{
					case LMX9830_READY:
						BTStatus.haveBT=true;
						break;
					case SPP_INCOMING_LINK_ESTABLISHED:
						BTStatus.linkEstablished=true;
						memcpy(BTStatus.remoteAddress,btPacket->data,6);
						BTStatus.localPort=btPacket->data[6];
						break;
					case SPP_LINK_RELEASED:
						BTStatus.linkEstablished=false;
						break;
					case SPP_TRANSPARENT_MODE:	// LMX9830 left transparent mode (probably due to link loss)
						break;
					default:
						if (BTStatus.unknTelCnt!=255)
						{
							BTStatus.unknTelCnt++;
						}
						break;
				}
			case ADEUNIS_CFM:
			case ADEUNIS_REQ:
			case ADEUNIS_RES:
			default:
				if (BTStatus.unknTelCnt!=255)
				{
					BTStatus.unknTelCnt++;
				}
				break;
		}				
	}
}

		volatile uint16_t xmin,xres;
/*!	\brief	Checks for Time Flags

	\note Be sure to check and clear all Flags even if you don't
	use them to avoid unnecessary calls of this function
*/
static void DoTflags(void)
{
	if (tFlags & TFLAG_SEC) {	// Do secondly stuff
		tFlags&=~TFLAG_SEC;	// Clear second-flag
		if (dFlags&DFLAG_ADC)
		{
			ADCcli();
			if (secCnt!=15)
			{
				hprintf_P(PSTR("%2d  "),secCnt);
				for (int i=0;i<5;i++)
					hprintf_P(PSTR("i:%d %4d/%4d~%4d  "),i,ADCDataMinTmp.array[i],ADCDataMaxTmp.array[i],ADCDataMaxTmp.array[i]-ADCDataMinTmp.array[i]);
				hprintf_P(PSTR("\n"));
			}
			ADCsei();
		}
		ADCResetMinMax();
		if (displayTimer==0 && config.pageTime!=0) {	// Every pageTime seconds
			ShowScreen(SCREENSTEPAUTO,0);	// Advance to next screen
			displayTimer=config.pageTime;
		} else
		{
			ShowScreen(SCREENREFRESH,0);
		}
		displayTimer--;
		Stackchk();
		switch (secCnt) {
/*			case 0:
				// standard display
				showScreen(SCREENREFRESH);
				break;*/
			case 1:
				displayValues.litersTemp=0.0;
				displayValues.nlitersTemp=0;
				discardFlightTime=true;	// Discard the first measurement
				SensorPowerOn();
				break;
			case 10:
				if (GetSig1())	// Alert #1 is active
				{
					if (SoundsOK() || (config.sndMask&SNDMASK_ALERT1ALWAYS))
					{
						PlaySequence(sndSig1);	// Play sound
					}
				}
				break;
			case 15:
				if (config.spontaneousReportVolume>0.0)
				{
					if (abs(displayValues.liters-lastReportedVolume)>=config.spontaneousReportVolume)
					{
						if (Tbub1IsEmpty())	// Send only if buffer empty to prevent overflow
						{
							hprintf_P(PSTR("\nTime:%d:%02d\nVolume: %f%c\nSensor Temp: %2.1fC\n"),hrCnt,minCnt,displayValues.liters,displayValues.literUnitChar,displayValues.tempSemi/2.0);
							lastReportedVolume=displayValues.liters;
						}
					}
				}
				break;
			case 20:
				if ((minCnt==0) && GetSig2())	// Alert #2 is active
				{
					if (SoundsOK() || (config.sndMask&SNDMASK_ALERT2ALWAYS))
					{
						PlaySequence(sndSig2);	// Play sound
					}
				}
				break;
			case 59:	// final checks
				if (displayValues.nlitersTemp<10) {
					displayValues.not10ErrorChar='<';	// no or not enough values received
				}
				else if (displayValues.nlitersTemp>10) {
					displayValues.not10ErrorChar='>';	// to many values received. probably a defective sensor power switch
				}
				if (sensorDead)	// We didn't receive a single telegram in the last minute
				{
					displayValues.sensorDeadChar='S';
					displayValues.liters=0;
					displayValues.tempSemi=0;
				}
				sensorDead=true;	// Set the flag, it will be deleted on the first telegram from the sensor
				SensorPowerOff();	// ensure a sensor reset
				break;
			}
		if (tFlags & TFLAG_MIN)	// Do minutely stuff
		{
			tFlags&=~TFLAG_MIN;
			if (tFlags & TFLAG_HOUR) {	// do hourly stuff
				tFlags&=~TFLAG_HOUR;	// clear hour-flag
				if (SoundsOK())
				{
					PlaySequence(sndBigBenSeq);
				}
				ResetDisplayFlags();
				displayValues.litersPerHour=displayValues.liters-displayValues.litersLastHour;
				displayValues.litersLastHour=displayValues.liters;
				if (tFlags & TFLAG_DAY) {	// do daily stuff
					tFlags&=~TFLAG_DAY;	// clear day-flag
					displayValues.litersToday=displayValues.liters-displayValues.litersYesterday;
					displayValues.litersYesterday=displayValues.liters;
				}
			}
			if (tFlags&TFLAG_DAY)	// We have no daily jobs until now
			{
				tFlags&=~TFLAG_DAY;	// Just reset the flag else this function would be called every 10ms!
			}
		}
		if (LCDGetBrightness()>0)	// LCD backlight is on
		{
			if (backlightOnTime<255)
			{
				backlightOnTime++;
				if ((backlightOnTime>config.backlightOnTime) && !backlightAlwaysOn)	// Backlight is on long enough now!
				{
					LCDSetBrightness(0);	// Switch it off
					backlightOnTime=0;		// And reset the on-time
				}
			}
		}
	}			
}

/*!	\brief	Checks for Key Flags

	\note Be sure to check and clear all Flags even if you don't
	use them to avoid unnecessary calls of this function
*/
static void DoKFlags(void)
{
	static bool	skipKey;
	
	displayTimer=config.pageTimeK;
	if (kflags&KFLAG_KEYPRESSED)	// The key has just been pressed
	{
		kflags&=~KFLAG_KEYPRESSED;	// Reset the flag
		if (LCDGetBrightness()==0)	// If the backlight was off
		{
			LCDSetBrightness(config.LCDBright);	// Switch it on now
			skipKey=true;	// And skip the next key event
		}
		backlightOnTime=0;
	}
	if (kflags&KFLAG_KEYPRESSEDSHORT)	// The key has been pressed for a short time and released
	{
		kflags&=~KFLAG_KEYPRESSEDSHORT;	// Reset the flag
		if (skipKey)	// The keypress has been marked for skipping
		{
			skipKey=false;
		} 
		else
		{
			ShowScreen(SCREENSTEPMANU,0);	// Not skipped, show the next screen
		}
	}
	if (kflags&KFLAG_KEYPRESSEDLONG)	// The key has been pressed for a long time
	{
		kflags&=~KFLAG_KEYPRESSEDLONG;
		LCDSetBrightness(255);	// give a signal to the user
		delay50ms();
		LCDSetBrightness(0);
		delay50ms();
		LCDSetBrightness(config.LCDBright);
		backlightAlwaysOn=!backlightAlwaysOn;	// Always on is inverted
		skipKey=false;
	}
	if (kflags&KFLAG_KEYRELEASED)	// No action for now...
	{
		kflags&=~KFLAG_KEYRELEASED;	// Just reset the flag for now!
	}
}

/*!	\brief Delay for a multiple of 10ms

	This function allows delays up to the range
	of delay_t controlled by the timer interrupt.<br>
	delay_t should be an unsigned integer type.<br>
	delaycnt must be decremented in the 100Hz-Interrupt.<br>
	This function of course takes care of the Watchdog and puts
	the CPU into sleep mode until the time expires.
	\param delay10ms	Delay in multiple of 10ms
	\note This function is not reentrant since there is only one delaycnt!
	The timer interrupt has to be enabled when calling this function
	else it will result in an infinite loop.
	Not even the watchdog would take you out!<br>
	The granularity of this function is 10ms, i.e. called
	with a value of 10, it may delay for 90..100ms.
*/
void	delay10ms(delay_t delay10ms)
{
	delaycnt=delay10ms;
	while (delaycnt)
	{
		wdt_reset();
		set_sleep_mode(SLEEP_MODE_IDLE);
		sleep_enable();
		sleep_cpu();
		sleep_disable();
	}
}

/*!	\brief	Sets the analog Instrument display
	\param	perCent	The value in a range from 0.0 to 100.0.
		Exceeding values will be trimmed correctly.
*/
void	SetAnalogPerCent(double perCent)
{
	if (perCent>100.0)	// Limit percent to 0..100
	{
		perCent=100.0;
	} else if (perCent<0.0)
	{
		perCent=0.0;
	}
	OCR1CL= ~(uint8_t)(perCent*config.instCal/100.0+0.5);	// Set the analog value
}

void HBarTest(void)
{
	LCDClear();
	LCDPrintLine_P(0,PSTR("Test"));
	for (uint8_t i=0;i<=100;i++)
	{
		LCDHBarGraph(3,i);
		delay10ms(10);
	}
	LCDHBarGraph(3,0);
	delay10ms(100);
	LCDHBarGraph(3,1);
	delay10ms(100);
	LCDHBarGraph(3,99);
	delay10ms(100);
	LCDHBarGraph(3,100);
	delay10ms(100);
	LCDClear();
}

void	VBarTestPrint()
{
	LCDPrintLine_P(0,PSTR("%c"),lcdVbarChar[3]);
	LCDPrintLine_P(1,PSTR("%c"),lcdVbarChar[2]);
	LCDPrintLine_P(2,PSTR("%c"),lcdVbarChar[1]);
	LCDPrintLine_P(3,PSTR("%c"),lcdVbarChar[0]);
}

void VBarTest(void)
{
	LCDClear();
	for (uint8_t i=0;i<=32;i++)
	{
		LCDBuildVBarChars(i);
		VBarTestPrint();
		delay10ms(10);
	}
	LCDBuildVBarChars(0);
	VBarTestPrint();
	delay10ms(100);
	LCDBuildVBarChars(1);
	VBarTestPrint();
	delay10ms(100);
	LCDBuildVBarChars(31);
	VBarTestPrint();
	delay10ms(100);
	LCDBuildVBarChars(32);
	VBarTestPrint();
	delay10ms(100);
	LCDClear();
}

/*!	\brief The Main Program.
*/
int main(void)
{
/*	volatile char	s[99];
	volatile int	i;
	i=-9;
	sprintf(s,"% 5d:",i);
	asm("nop");*/
	
	Init();	// Initialize the Hardware
	ShowScreen(SCREENABSPAGE,SCREENABSPAGE_VERSION);	// Show version screen
	PlaySequence(sndSuccessSeq);
	displayTimer=config.pageTime;
/*	if (GetKeyState())	// Key is pressed during power up
	{
		FTest485();
		for(;;);	// Watchdog reset
	}*/
	set_sleep_mode(SLEEP_MODE_IDLE);
	while (1)  {
		wdt_reset();
		LEDGreenOn();
		sleep_enable();
		sleep_cpu();	// sleep until something happens...
		sleep_disable();
		LEDGreenOff();

		// Check flags set by interrupts
		if (cFlags)	// Communication flags
		{
			DoCflags();
		}
		if (tFlags)	// Time flags
		{
			DoTflags();
		}
		if (kflags)	// Key flags
		{
			DoKFlags();
		}
		/*	Set error message. This has to be done in ascending priority
			to ensure that the most urgent message is displayed.
			The most important error has to be the last one
			in the following code. Only this one will display. */
		double unregMax, unregMin;
		
		unregMax=UnregFromADC(ADCDataMax.named.ADC_Unreg);
		unregMin=UnregFromADC(ADCDataMin.named.ADC_Unreg);
		if ((unregMax-unregMin)>9.5)
		{
			snprintf_P(displayError,sizeof(displayError),PSTR("Ripple HIGH (C10?)! (%f-%f=%f)"),unregMax,unregMin,unregMax-unregMin);
			displayValues.error=displayError;
		}
		if (unregMin<15)
		{
			snprintf_P(displayError,sizeof(displayError),PSTR("Unreg LOW! (%f)"),unregMin);
			displayValues.error=displayError;
		}
		if (unregMax>30.0)
		{
			snprintf_P(displayError,sizeof(displayError),PSTR("Unreg HIGH! (%f)"),unregMax);
			displayValues.error=displayError;
		}
	}
}
