﻿/*! \file config.c

	\brief	Routines for handling Configuration Data
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <avr/eeprom.h>
#include <stdio.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include "display.h"
#include "config.h"
#include "uart.h"
#include "dcf77.h"
#include "delay.h"

//! The Configuration. It is read on startup from the EEPROM.
confStruct_t	config;

/*!	\brief This Struct will be stored in the EEPROM and is contained in
	the Production File.
	
	'New' devices are programmed using the production file which also can
	write the default EEPROM content and the fuses.
*/
confStruct_t	EEMEM defaultConfig=
{
	.magic=CONF_MAGIC,	// Magic number to detect unprogrammed devices
	.displayType=DISPLAY_4x20,	// The type of display used in our configuration
	.cisternType=CIST_CONST_AREA,
	.cistAreaOrDiameter=3.0,	// The surface area in a constant-surface-type cistern
	.cistHeight=2.8,	// The height of the sensor over ground
	.cistCapacity=6000.0,	// Capacity in liters
	.invDCF77=0,
	.LCDV0=25,	// Initial control voltage for the LC-Display
	.LCDBright=255,	// I love light! And I love power dissipation!
	.P3VoltsPerLSB=ADCVoltsPerLSB/dividerP3,	// Volts per LSB of the AD-Converter for 3V supply
	.P5VoltsPerLSB=ADCVoltsPerLSB/dividerP5,	// Volts per LSB of the AD-Converter for 5V supply
	.UnregVoltsPerLSB=ADCVoltsPerLSB/dividerUnreg,	// Volts per LSB of the AD-Converter for Unreg supply
	.SWUnregVoltsPerLSB=ADCVoltsPerLSB/dividerSWUnreg,	// Volts per LSB of the AD-Converter for SWUnreg supply
	.backlightAlwaysOn=true,	// I mentioned before, I love light!
	.backlightOnTime=5,	// Backlight is switched off backlightOnTime seconds after the key has been pressed unless backlightAlwaysOn is true
	.pageTime=4,	// Every four seconds the display page changes
	.pageTimeK=60,	// After the key is pressed, the display page remains for pageTimeK seconds before automatic stepping is reestablished
	.signalOutputs[0].activeBelow=0,
	.signalOutputs[0].activeAbove=65535,
	.signalOutputs[1].activeBelow=0,
	.signalOutputs[1].activeAbove=65535,
	.sndHourFrom=9,	// Sounds are only played after 9am
	.sndHourTo=21,	// Sounds are only played until 9pm
	.sndMask=0,	// No special sound treatment by default
	.instCal=117,	// Duty cycle for analog instrument FSR. 138 (255-117) gives 100uA
	.useRts=true,	// Use hardware handshake by default
	.spontaneousReportVolume=0.0,
};

/*!	\brief	Loads default config values into conf-struct and stores it in non volatile Memory

	This function is mostly useful during development but I recommend to leave it functional.
	You can omit this routine if you want to save about 300 bytes of flash but it is
	quite useful if you change the config struct and change CONF_MAGIC. It will reprogram
	the EEPROM with default values then.
	Especially in devices already in the field it would lead to unpredictable results
	if the configuration memory format would not be compatible with the current version.
	\note Keep the values here in sync with the values in defaultConfig if you don't
	like surprises!
*/
static void DefaultConfig(void)
{
	delay1s();	// If you use the production file, EEPROM write may be interrupted by a reset if we don't a delay here!
	config.magic=CONF_MAGIC;	// Magic number to detect unprogrammed devices
	config.displayType=DISPLAY_4x20;	// The type of display used in our configuration
	config.cisternType=CIST_CONST_AREA; 
	config.cistAreaOrDiameter=3.0;	// The surface area in a constant-surface-type cistern
	config.cistHeight=2.8;	// The height of the sensor over ground
	config.cistCapacity=6000.0;	// Capacity in liters
	config.invDCF77=0;
	config.LCDV0=25;	// Initial control voltage for the LC-Display
	config.LCDBright=255;	// I love light! And I love power dissipation!
	config.P3VoltsPerLSB=ADCVoltsPerLSB/dividerP3;	// Volts per LSB of the AD-Converter for 3V supply
	config.P5VoltsPerLSB=ADCVoltsPerLSB/dividerP5;	// Volts per LSB of the AD-Converter for 5V supply
	config.UnregVoltsPerLSB=ADCVoltsPerLSB/dividerUnreg;	// Volts per LSB of the AD-Converter for Unreg supply
	config.SWUnregVoltsPerLSB=ADCVoltsPerLSB/dividerSWUnreg;	// Volts per LSB of the AD-Converter for SWUnreg supply
	config.backlightAlwaysOn=true;	// I mentioned before, I love light!
	config.backlightOnTime=5;	// Backlight is switched off backlightOnTime seconds after the key has been pressed unless backlightAlwaysOn is true
	config.pageTime=4;	// Every four seconds the display page changes
	config.pageTimeK=60;	// After the key is pressed, the display page stays for pageTimeK seconds before automatic stepping is reestablished
	for (uint8_t i=0;i<NUM_SIGNAL;i++)
	{
		config.signalOutputs[i].activeBelow=0;
		config.signalOutputs[i].activeAbove=65535;
	}
	config.sndHourFrom=9;	// Sounds are only played after 9am
	config.sndHourTo=21;	// Sounds are only played until 9pm
	config.sndMask=0;	// No special sound treatment by default
	config.instCal=138;	// Duty cycle for analog instrument FSR. 138 gives 100uA
	config.useRts=true;	// Use hardware handshake by default
	config.spontaneousReportVolume=0.0;	// No spontaneous reports
	WriteConfig();
}

bool configDirty=false;	///< Set to true by routines affecting configuration if config memory needs to be saved to the EEPROM

/*!	\brief	Checks if sounds may currently be played

	"Currently" means at the actual time-of-day.
	Check this function before you play sounds that should
	not be always enabled.
	\return	true if sounds are ok, false if not
*/
bool	SoundsOK(void)
{
	if ((hrCnt>=config.sndHourFrom) && (hrCnt<config.sndHourTo))
	{
		return true;
	} 
	else
	{
		return false;
	}
}

/*!	\brief	Reads the Config-Struct from EEPROM into config.

	Only the <i>config</i>-struct in RAM will be used. Changes in <i>config</i> will only
	become permanent if you use a write-command. There is a <i>configDirty</i>-flag
	that is set by every routine that changes <i>config</i> and is used to remind the
	user to execute a write-command.
*/
void ReadConfig(void)
{
	eeprom_read_block(&config,&defaultConfig,sizeof(confStruct_t));	// Copy the default config to the config hold in RAM
	if (config.magic!=CONF_MAGIC)	// no configuration data in EEPROM
	{
		DefaultConfig();	// Obsolete since we use EEMEM-Struct
	}
	configDirty=false;	// Configuration is valid from now
}

/*!	\brief	Writes the <i>Config</i>-Struct to EEPROM
*/
void WriteConfig(void)
{
	cli();	// Do not disturb!
	eeprom_write_block(&config,&defaultConfig,sizeof(confStruct_t));	// Copy the configuration data from RAM to EEPROM
	eeprom_busy_wait();
	configDirty=false;	// Configuration is valid now since RAM and EEPROM are now the same
	sei();
}

/*!	\brief	DumpConfig Helper Function for Optocoupler outputs
	\param	number The Number of the Signal Output (Zero-based)
*/
void DumpConfigSignal(uint8_t number)
{
	hprintf_P(PSTR("Signal Optocoupler %d is "),number+1);
	if (config.signalOutputs[number].activeBelow>0 && config.signalOutputs[number].activeAbove<65535)
	{
		hprintf_P(PSTR("ON below %u and above %u liters\n"),config.signalOutputs[number].activeBelow,config.signalOutputs[number].activeAbove);
	} else
	if (config.signalOutputs[number].activeBelow>0)
	{
		hprintf_P(PSTR("ON below %u liters\n"),config.signalOutputs[number].activeBelow);
	} else
	if (config.signalOutputs[number].activeAbove<65535)
	{
		hprintf_P(PSTR("ON above %u liters\n"),config.signalOutputs[number].activeAbove);
	} else
	{
		hprintf_P(PSTR("never ON\n"));
	}
}

/*!	\brief	Lists the current configuration values

	Prints a verbose explanation of the configuration data
	to the host.
*/
void DumpConfig(void)
{
	hprintf_P(PSTR("\nCurrent configuration:\n"));
	hprintf_P(PSTR("Display Type: "));
	switch(config.displayType)
	{
		case DISPLAY_4x20:
			hprintf_P(PSTR("4x20\n"));
			break;
		case DISPLAY_2x16:
			hprintf_P(PSTR("2x16\n"));
			break;
		case DISPLAY_ANALOG:
			hprintf_P(PSTR("analog\n"));
			break;
		default:
			hprintf_P(PSTR("(invalid)\n"));
			break;
	}
	hprintf_P(PSTR("Analog Meter FSR: %d\n"),config.instCal);
	hprintf_P(PSTR("Cistern Type: "));
	switch (config.cisternType) {
		case CIST_CONST_AREA:
			hprintf_P(PSTR("Constant Area\n"));
			hprintf_P(PSTR("Cistern Area: %.3f sqm\n"),config.cistAreaOrDiameter);
			break;
		case CIST_HORIZ_CYL:
			hprintf_P(PSTR("Horizontal Cylinder\n"));
			hprintf_P(PSTR("Cistern Diameter: %.2f m\n"),config.cistAreaOrDiameter);
			break;
		case CIST_SPHERE:
			hprintf_P(PSTR("Sphere\n"));
			hprintf_P(PSTR("Cistern Diameter: %.2f m\n"),config.cistAreaOrDiameter);
			break;
		default:
			hprintf_P(PSTR("unknown\n"));
	}
	hprintf_P(PSTR("Capacity: %.0 l\n"),config.cistCapacity);
	hprintf_P(PSTR("Sensor distance to Ground: %.3f m\n"),config.cistHeight);
	hprintf_P(PSTR("Invert DCF77: "));
	switch(config.invDCF77)
	{
		case 0:
			hprintf_P(PSTR("no\n"));
			break;
		case DCF77_PINMASK:
			hprintf_P(PSTR("yes\n"));
			break;
		default:
			hprintf_P(PSTR("(invalid)\n"));
			break;
	}
	DumpConfigSignal(0);
	DumpConfigSignal(1);
	hprintf_P(PSTR("LCD Backlight Brightness: %d\n"),config.LCDBright);
	hprintf_P(PSTR("LCD V0: %d\n"),config.LCDV0);
	hprintf_P(PSTR("Backlight always on: %d\n"),config.backlightAlwaysOn);
	hprintf_P(PSTR("Backlight on time: %d\n"),config.backlightOnTime);
	hprintf_P(PSTR("Display stepping delay: %ds\n"),config.pageTime);
	hprintf_P(PSTR("Manual stepping switchback delay: %ds\n"),config.pageTimeK);
	hprintf_P(PSTR("P3 mVolts per LSB: %.3f\n"),config.P3VoltsPerLSB*1000.0);
	hprintf_P(PSTR("P5 mVolts per LSB: %.3f\n"),config.P5VoltsPerLSB*1000.0);
	hprintf_P(PSTR("Unreg mVolts per LSB: %.3f\n"),config.UnregVoltsPerLSB*1000.0);
	hprintf_P(PSTR("SWUnreg mVolts per LSB: %.3f\n"),config.SWUnregVoltsPerLSB*1000.0);
	hprintf_P(PSTR("Sounds are played from %d:00 to %d:00\n"),config.sndHourFrom,config.sndHourTo);
	if (config.sndMask & SNDMASK_ALERT1ALWAYS)
	{
		hprintf_P(PSTR("Alert 1 is played always\n"));
	}
	if (config.sndMask & SNDMASK_ALERT2ALWAYS)
	{
		hprintf_P(PSTR("Alert 2 is played always\n"));
	}
	if (config.sndMask & SNDMASK_HOURBELL)
	{
		hprintf_P(PSTR("Big Ben is played every hour\n"));
	}
	hprintf_P(PSTR("Serial use RTS/CTS handshake: %s\n"),config.useRts?"Yes":"No");
}