;***************************************************************************
;
;          File: Sensor.asm
;        Author: Robert Loos
;
;   Description: Firmware for sensor unit
;
;   Hardware: AT90S2313 based
;
;
;-------------------------------[ History ]---------------------------------
;
;  2008-04-24 Initial revision. VERSION = 1
;
;
;---------------------------------------------------------------------------
; (c) Copyright 2009 Robert Loos
;***************************************************************************

.INCLUDE	"tn2313def.inc"

; Special version: Set this to 1 to use readable ASCII telegrams
; Values are sent in Hex and with 8n2 then.
; The bootloader is non-functional in this Version!
; Default: 0
.equ	USEHEX		=0

; Definitions for US-transmitter
.equ	PORTQ		=PORTD
.equ	BITQ		=PD4


;******************* variables
.DSEG
deadTime:	.BYTE	1

;******************* code
.CSEG
.ORG	0x0000
		rjmp	bootld		; Reset
		rjmp	RESET		; INT0
		rjmp	RESET		; INT1
		rjmp	T1CAPT		; Timer1 CAPT1
		rjmp	RESET		; Timer1 COMP1
		rjmp	T1OVF		; Timer1 OVF1
		rjmp	RESET		; Timer0 OVF0
		rjmp	RESET		; Uart RX
		rjmp	RESET		; Uart UDRE
		rjmp	RESET		; Uart TX
		rjmp	RESET		; ANA_COMP
		rjmp	RESET		; PCINT
		rjmp	RESET		; Timer1 COMPB
		rjmp	RESET		; Timer0 COMPA
		rjmp	RESET		; Timer0 COMPB
		rjmp	RESET		; USI Start
		rjmp	RESET		; USI Overflow
		rjmp	RESET		; EE Ready
		rjmp	RESET		; WDT Overflow

RESET:
		ldi		r16,0xdf
		out		spl,r16
		rcall	init_IO
		rcall	delay100ms	; allow analog cirquit to stabilize
		ldi		r16,30			; setup dead time for about 30cm
		sts		deadTime,r16

LOOP:
		rcall	measureTime
		rcall	sendTime
		rcall	measureTemp
		rcall	sendTemp
		rcall	getSerial
		rjmp	loop

;*************************************
; Subroutines
;*************************************
; Reserved registers:
; R0/R1 receive the traveling time (L/H) of the US-pulse

init_IO:	; set port direction and usage
		ldi		r16,0b11111100
		out		ddrb,r16
		ldi		r16,0b10111110
		out		ddrd,r16
		sbi		portd,pd0		; enable pullup for RxD
.if	USEHEX==0
		ldi		r16,0b00000000	; UART to 9e2
		out		UCSRA,r16
		ldi		r16,0b00011100	; RXCIE TXCIE UDRIE RXEN TXEN UCSZ2 RXB8 TXB8
		out		UCSRB,r16
		ldi		r16,0b00101110	; - UMSEL UPM1 UPM0 USPS UCSZ1 UCSZ0 UCPOL
		out		UCSRC,r16
.else
		ldi		r16,0b00000000	; UART to 8n1
		out		UCSRA,r16
		ldi		r16,0b00011000	; RXCIE TXCIE UDRIE RXEN TXEN UCSZ2 RXB8 TXB8
		out		UCSRB,r16
		ldi		r16,0b00001110	; - UMSEL UPM1 UPM0 USPS UCSZ1 UCSZ0 UCPOL
		out		UCSRC,r16
.endif
		ldi		r16,23		; 9600 baud
		out		ubrrl,r16
		clr		r16
		out		ubrrh,r16
		ldi		r16,(1<<AIN1D) | (1<<AIN0D)	; disable digital input pins for comparator
		out		DIDR,r16
		ret

delay100ms:
		push	r18
		ldi		r18,100
		rcall	delay1ms
		dec		r18
		brne	pc-2
		pop		r18
		ret

delay1s:
		push	r19
		ldi		r19,50
		rcall	delay100ms
		dec		r19
		brne	pc-2
		pop		r19
		ret

;******************************
; timer 1 interrupt
T1OVF:
		push	r2
		in		r2,SREG
		clr		r0
		clr		r1
		out		SREG,r2
		pop		r2
		reti
T1CAPT:
		push	r2
		in		r2,SREG
		in		r0,ICR1L
		in		r1,ICR1H
;	push	r16
;	push	r17
;	in		r16,PORTB
;	ldi		r17,(1<<2)
;	eor		r16,r17
;	out		PORTB,r16
;	pop		r17
;	pop		r16
		out		SREG,r2
		pop		r2
		reti

;******************************
; do one measurement
; r20 is used for clock multiplicator (1 or 8)
measureTime:
		;setup timer 1
		ldi		r16,0b00000000
		out		TCCR1A,r16	; disable OC1 and PWM
		ldi		r16,0b01000001
		out		TCCR1B,r16	; clk/1
		ldi		r20,1		; clock multiplicator for sendTime
		ldi		r18,2		; dead time
		rcall	measureTime1
		tst		r0
		brne	measureTimeExit
		tst		r1
		brne	measureTimeExit
		ldi		r16,0b01000010		; timer overflow, try with clk/8
		out		TCCR1B,r16	; clk/8
		ldi		r20,8
		rcall	measureTime1
measureTimeExit:
		ret

; do one measurement
; r18 contains the desired dead time in ms
measureTime1:
		ldi		r16,0b00000000
		out		TIMSK,r16	; disable timer1 overflow and input capture
		clr		r16			; reset timer1
		out		TCNT1H,r16
		out		TCNT1L,r16
		cli
		ldi		r17,8		; generate a packet of 8 40kHz-pulses
pulseloop:
		sbi		PORTQ,BITQ
		ldi		r16,14
		dec		r16
		brne	pc-1
		cbi		PORTQ,BITQ
		ldi		r16,14
		dec		r16
		brne	pc-1
		dec		r17
		brne	pulseloop

deadTimeLoop:
		in		r16,TIFR	; clear input capture flag
		sbr		r16,(1<<ICF1)
		out		TIFR,r16

		ldi		r16,0b10001000
		out		TIMSK,r16	; enable timer1 overflow and input capture
		sei
		ldi		r16,0b00100000
		out		MCUCR,r16
deadLoop:
		sleep
; traveling time is now in r0/r1
; or 00/00 if nothing was received
		mov		r16,r0
		or		r16,r1		; result zero (timeout)?
		breq	deadLoopEx
		lds		r16,deadTime
		cp		r1,r16
		brlo	deadLoop
deadLoopEx:
		ldi		r16,0b00000000
		out		TIMSK,r16	; no more interrupts
		ret

	; send the time in r0/r1 over the uart
sendTime:
		clr		r18			; checksum
		ldi		r17,1		; set bit 9
		ldi		r16,'T'		; id-byte for time
		rcall	txByte
		clr		r17			; bit9=0
.if	USEHEX==0
		mov		r16,r1		; high-byte
		rcall	txByte
		mov		r16,r0		; low-byte
		rcall	txByte
		mov		r16,r20		; multiplicator
		rcall	txByte
		lds		r16,deadTime
		rcall	txByte
		mov		r16,r18		; finally the checksum
		rcall	txByte
		inc		r17			; set 9th bit
		clr		r16
		rcall	txByte		; send 0
.else
		rcall	txSpace
		mov		r16,r1		; high-byte
		rcall	txHex
		mov		r16,r0		; low-byte
		rcall	txHexSpace
		mov		r16,r20		; multiplicator
		rcall	txHexSpace
		lds		r16,deadTime
		rcall	txHex
		rcall	txLF
.endif
		rjmp	txEnd

	; send the temperature over the uart
	; r0 is expected to contain the temp as signed char in 0.5C
	; r20 is expected to contain the raw A/D-value
sendTemp:
		clr		r18			; checksum
		ldi		r17,1		; set bit 9
		ldi		r16,'t'		; id-byte for temp
		rcall	txByte
		clr		r17			; bit9=0
.if	USEHEX==0
		mov		r16,r0		; temperature in 0.5C
		rcall	txByte
		mov		r16,r20		; raw A/D-value
		rcall	txByte
		mov		r16,r18		; finally the checksum
		rcall	txByte
		inc		r17			; set 9th bit
		clr		r16
		rcall	txByte		; send 0
.else
		rcall	txSpace
		mov		r16,r0		; temperature in 0.5C
		rcall	txHexSpace
		mov		r16,r20		; raw A/D-value
		rcall	txHex
		rcall	txLF
.endif
		rjmp	txEnd

measureTemp:
		; set timer1 to fast-pwm 8 bit
		clr		r16
		out		TIMSK,r16	; no interrupts
		ldi		r16,(1<<COM1A1) | (1<<WGM10)
		out		TCCR1A,r16
		ldi		r16,(1<<WGM12) | (1<<CS10)
		out		TCCR1B,r16
		clr		r16
		out		ACSR,r16	; switch on analog comparator
		ldi		r17,0x80		; init SAR-bitmask
		clr		r20			; init SAR-value
mtLoop:
		clr		r16			; set OCR to SAR-value, clear timer
		add		r20,r17		; adjust SAR-Value
		out		OCR1AH,r16
		out		OCR1AL,r20
		out		TCNT1H,r16
		out		TCNT1L,r16
		rcall	delay100ms	; wait for the rc-voltage to stabilize
		sbi		PORTB,pb4		; switch on NTC
		rcall	delay1ms	; NTC-voltage settling time
		sbic	ACSR,ACO	; NTC-voltage > DAC-voltage?
		sub		r20,r17		; no, try the next smaller step
		cbi		PORTB,pb4		; switch off NTC
		lsr		r17
		brne	mtLoop
		ldi		r16,(1<<ACD)
		out		ACSR,r16	; switch off analog comparator
		ldi		zl,LOW(tempLUT*2)
		ldi		zh,HIGH(tempLUT*2)
		add		zl,r20
		adc		zh,r17		; r17 happens to be zero here, no need to clear before
		lpm					; r0 now contains the real temperature
		ret

; temperature lookup table
tempLUT:
; temperature values as signed char in 0.5-units
		.db \
			127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127, \
			127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127, \
			127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,125, \
			124,123,121,120,118,117,116,115,113,112,111,110,108,107,106,105, \
			104,103,102,101,100,99,97,96,95,94,93,92,92,91,90,89, \
			88,87,86,85,84,83,82,81,81,80,79,78,77,76,75,75, \
			74,73,72,71,71,70,69,68,67,67,66,65,64,64,63,62, \
			61,61,60,59,58,58,57,56,55,55,54,53,53,52,51,50, \
			50,49,48,48,47,46,45,45,44,43,43,42,41,40,40,39, \
			38,38,37,36,35,35,34,33,33,32,31,31,30,29,28,28, \
			27,26,26,25,24,23,23,22,21,20,20,19,18,17,17,16, \
			15,14,14,13,12,11,11,10,9,8,7,7,6,5,4,3, \
			3,2,1,0,-1,-2,-3,-3,-4,-5,-6,-7,-8,-9,-10,-11, \
			-12,-13,-14,-15,-16,-17,-18,-19,-20,-21,-23,-24,-25,-26,-27,-29, \
			-30,-31,-33,-34,-35,-37,-38,-40,-42,-43,-45,-47,-49,-51,-53,-55, \
			-57,-59,-62,-65,-68,-71,-74,-78,-82,-87,-92,-99,-107,-118,-128,-128

; if a character is received on the Usart
; it is used as dead time (compared against high byte of timer1)
; the step is thus 69us or approx. 11.8mm in distance
getSerial:
		in		r16,UCSRA
		sbrs	r16,RXC			; did we receive sonething?
		ret						; no
		in		r16,UDR
		sts		deadTime,r16
		ret

txHexSpace:
		rcall	txHex
txSpace:
		ldi		r16,' '
		rjmp	txByte

txLF:
		ldi		r16,10	; Line feed character
		rjmp	txByte

; Converts r16 to ASCII-HEX and sends it
txHex:	push	r16
		swap	r16
		rcall	txHexNibble
		pop		r16
txHexNibble:
		andi	r16,$0f
		subi	r16,-'0'
		cpi		r16,'9'+1
		brcs	txHexSend
		subi	r16,-7
txHexSend:
		rjmp	txbyte


; *******************
; Firmware updates
.org 0x320		; must be a page boundry
bootldProtect:	; protected area starts here

delay50us:	; takes 50.18us including the rcall
		push r16		; 2
		ldi		r16,58	; 1
		dec		r16		; 1
		brne	pc-1	; 2 (1)
		pop r16			; 2
		ret				; 4

; delay 1ms
; this routine uses r15 to save sreg
; r15 is therefore changed
delay1ms:
		in		r15,sreg
		push	r17
		ldi		r17,20
		rcall	delay50us
		dec		r17
		brne	pc-2
		pop		r17
		out		sreg,r15
		ret


; the bootloader code
; timer1 is used for timeout
bootld:
		ldi		r16,0xdf		; Set stack pointer to highest RAM-address
		out		spl,r16
		rcall	delay1ms
		ldi		r16,0b11111100
		out		ddrb,r16
		ldi		r16,0b10111110
		out		ddrd,r16
		sbi		portb,pb5
		ldi		r16,0b00000001	; timer1 clk/1
		out		TCCR1B,r16		; will overflow after 17.8ms
		ldi		r16,0b00000000	; UART to 9e2
		out		UCSRA,r16
		ldi		r16,0b00011100	; 9-Bit mode
		out		UCSRB,r16
		ldi		r16,0b00101110
		out		UCSRC,r16
		ldi		r16,23		; 9600 baud
		out		ubrrl,r16
		clr		r16
		sbi		portd,pd0		; enable pullup for RxD
		out		ubrrh,r16
		out		TCCR1A,r16
		rcall	delay1ms		; allow supply to stabilize
		ldi		r17,0x01
		ldi		r16,'*'
		rcall	txByte
		clr		r17
		rcall	txByte		; a single char is its own checksum!
		ldi		r17,0x01
		clr		r16
		rcall	txByte		; character 0 with bit 9=1
		rcall	txEnd
		rcall	flushRx
		rcall	rxByteTimed
		brcs	bootldExTimeout	; timeout
		tst		r18
		brne	bootldExRxErr	; some error
		tst		r17
		breq	bootldExBit9	; bit 9 not set
		cpi		r16,'#'
		brne	bootldExNoSharp	; wrong answer
		; ok, bootloader is active now
		ldi		r16,0b00000011	; timer0 clk/64
		out		TCCR1B,r16		; will overflow after 1.1s
bootldLoop:
		clr		r19			; init checksum
		rcall	rxByteTimed
		brcs	bootldExTimeout
		cpi		r16,'!'		; write command
		breq	bootldWrite
		cpi		r16,'?'		; read command
		breq	bootldRead
		; else exit
bootldExTimeout:	// The different txBytes are just for testing
bootldExRxErr:
bootldExBit9:
bootldExNoSharp:
bootldEx:
		cbi		portb,pb5
		rjmp	0x0013		; start main program


echoOk:
		ldi		r16,'+'
		rjmp	echoCont
echoError:
		ldi		r16,'-'
		rjmp	echoCont
echoProtected:
		ldi		r16,'P'
echoCont:
		rcall	delay1ms
		ldi		r17,1		; set bit 9
		rcall txByte
		clr		r17
		rcall	txByte
		ldi		r17,1
echoEx:	clr		r16					; zero-byte with bit9=1
		rcall	txByte
		rcall	txEnd
		rjmp	bootldLoop			; ready for next command


; receives one write-dataset
bootldWrite:
		rcall	readTilBit9
		brne	echoError
		cpi		r26,SRAM_START + 36		; telegram length ok?
		brne	echoError
		ldi		r26,low(SRAM_START+2)	; set X to data in sram
		ldi		r27,high(SRAM_START+2)
		lds		r31,SRAM_START			; get flash start address
		lds		r30,SRAM_START+1
		mov		r16,r30
		andi	r16,0x1f				; mask out address bits in page
		brne	echoError				; address is not on a page boundary
		ldi		r16,high(bootldProtect*2)	; need byte address here
		cpi		r30,low(bootldProtect*2)
		cpc		r31,r16
		brge	echoProtected			; do not modify bootloader itself!
		clr		r16
		cp		r30,r16
		cpc		r31,r16					; is it page 0?
		brne	notPage0
		ldi		r16,high(bootld-1)|0xc0	; yes, fix resetvector!
		sts		SRAM_START+3,r16
		ldi		r16,low(bootld-1)
		sts		SRAM_START+2,r16
notPage0:
		rcall	writePage
		rjmp	echoOk

; receives one read-dataset
bootldRead:
		rcall	readTilBit9
		brne	echoError
		cpi		r26,SRAM_START + 4	; telegram length ok?
		brne	echoError
		clr		r18					; init checksum
		ldi		r16,'='
		ldi		r17,0x01
		rcall	txByte				; signal comming data
		lds		r30,SRAM_START+1
		lds		r31,SRAM_START
		ldi		r19,32				;byte counter
bootldRdLoop:
		lpm		r16,Z+
		clr		r17
		rcall	txByte
		dec		r19
		brne	bootldRdLoop
		mov		r16,r18				; finally the checksum
		rcall	txByte
		clr		r16					; and the zero-byte
		ldi		r17,0x01			; with bit 9 set
		rcall	txByte
		rcall	txEnd
		rjmp	bootldLoop			; ready for next command

; reads until reception of a character
; with bit 9 set (which should be the cheksum)
; but no more than 48 bytes
; Z-flag is set on checksum ok
; Z-flag clear indicates error
readTilBit9:
		ldi		r26,SRAM_START	; X = ram start
		clr		r27
rtb9l:	; loop
		rcall	rxByteTimed
		brcs	readTilBit9ex	; timeout
		tst		r18
		brne	readTilBit9ex2	; error
		st		x+,r16
		cpi		r26,SRAM_START + 48
		breq	readTilBit9ex2	; more than 48 chars received (?)
		tst		r17
		breq	rtb9l		; bit 9 not set. some more to come...
		eor		r16,r19		; compare checksum
readTilBit9ex:
		rjmp	delay1ms	; to prevent transmitter overlap
readTilBit9ex2:
		clz		; clear zero flag for error indication
		rjmp	delay1ms	; to prevent transmitter overlap
bootldEnd:


; waits for a byte on the serial.
; timer1 is used for timeout
; and must be properly set
; the character is returned in r16
; bit 0 of r17 contains bit 9
; r18 contains error flags as in UCSRA
; carry-flag is set on timeout
; the eor-checksum is traced in r19
rxByteTimed:
;	clr r18
;	clc
;	ret	; for simulator test
sbi	portb,pb2	; give a visible signal
		clr		r16			; start timeout
		out		TCNT1H,r16
		out		TCNT1L,r16
		nop
		ldi		r16,(1<<TOV1)
		out		TIFR,r16
rxbt1:
		wdr
		in		r16,TIFR	; timeout?
		andi	r16,(1<<TOV1)
		brne	rxbttimeout	; -> exit
		sbis	UCSRA,RXC
		rjmp	rxbt1		; wait for char
		in		r18,UCSRA
		andi	r18,(1<<FE)|(1<<DOR)|(1<<UPE)	; mask out error flags
		in		r17,UCSRB	; get bit 9 into r17.0
		andi	r17,0x02	; mask out bit 0
		lsr		r17
		in		r16,UDR
		eor		r19,r16		; checksum
cbi	portb,pb2
		clc
		ret
rxbttimeout:
cbi	portb,pb2
		sec
		ret

; flushes uart receive buffer
; i.e. discard all received chars
flushRx:
		in		r16,udr
		sbic	UCSRA,RXC
		rjmp	flushRx
		ret

; Writes a page to Flash
; Z points to the Flash-address
; Z must be the beginning of a page
; X points to the source data wich must be
; 32 bytes (one page) long
writePage:
		push	r30
		push	r31
		ldi		r16,0b00000011	; page erase & store program memory enable
		out		SPMCSR,r16
		spm						; erase the page
wpLoop:
		ld		r0,X+
		ld		r1,X+
		ldi		r16,0b00000001	; store program memory enable
		out		SPMCSR,r16
		spm						; store R1:R0 in temporary buffer
		inc		r30				; inc Z (overflow cannot happen here)
		inc		r30
		mov		r16,r30
		andi	r16,0x1f
		brne	wpLoop
		pop		r31
		pop		r30
		ldi		r16,0b00000101	; page write & store program memory enable
		out		SPMCSR,r16
		spm
		ret

; send a byte in R16 over the uart
; we don't use interrupts here
; if UDR is busy, this routine blocks
; bit 0 in R17 contains the 9th bit
; R18 is intendet to create a checksum and is modified
txByte:
		sbi		PORTD,2		; enable line driver
		sbis	UCSRA,5
		rjmp	txByte		; wait until uart ready
		cbi		UCSRB,TXB8;	; copy 9th bit to TXB8
		sbrc	r17,0
		sbi		UCSRB,TXB8
		out		UDR,r16
		eor		r18,r16		; checksum
		ret

; wait until the transmitter is idle
; and switch off the line driver
txEnd:
		sbi		UCSRA,6
txEndL:
		sbis	UCSRA,6
		rjmp	txEndL
		cbi		PORTD,2		; disable line driver
		ret
