﻿/*!	\file usartcmd.c
	Handles commands from the serial interface

	\brief	Handles commands from the Host Interface
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <avr/interrupt.h>
#include "adc.h"
#include "adeunis.h"
#include "delay.h"
#include "display.h"
#include "uart.h"
#include "config.h"
#include "lcd.h"
#include "dcf77.h"
#include "ihex.h"
#include "ftest.h"
#include "usartcmd.h"
#include "sensors.h"

static const char helpCalibrate[];
static const char helpClearErrors[];
static const char helpDCFinv[];
static const char helpDebug[];
static const char helpDumpConf[];
static const char helpFTest[];
static const char helpGetSensorValues[];
static const char helpHelp[];
static const char helpReset[];
static const char helpSetBright[];
static const char helpSetCistern[];
static const char helpSetContr[];
static const char helpSetDisplay[];
static const char helpSetPTime[];
static const char helpSetPTimeK[];
static const char helpSetSignal[];
static const char helpSetSounds[];
static const char helpSetTime[];
static const char helpStatus[];
static const char helpUseRts[];
static const char helpUpdate[];
static const char helpVerify[];
static const char helpWrite[];
static const char helpDistance[];
static const char helpSetSpontaneous[];

#define MAXPARAMS	5
static char	*cmd;
static char	*params[MAXPARAMS];
static uint8_t	nParams;

/*!	\brief Prints error message about superfluous parameters
	\param n The number of <b>extra</b> parameters
*/
static void ExtraIgnored(uint8_t n)
{
	hprintf_P(PSTR("%d extra parameters ignored in command %s\n"),n,cmd);
}

/*!	\brief Prints error message about missing arguments
*/
static void MissingArg(void)
{
	hprintf_P(PSTR("Missing value in command %s\n"),cmd);
}

/*!	\brief Prints the Configuration Memory in a human readable form
	\return always 0
*/
static int8_t	CmdDumpConf(void)
{
	DumpConfig();	// Realized in config.c
	if (nParams>0)
		ExtraIgnored(nParams);
	hprintf_P(PSTR("Have a nice day!\n"));
	return 0;	// always succeeds
}

/*!	\brief Retrieves the Sensor Values

	Measured volume and temperature inside the cistern.
	\return always 0
*/
static int8_t	CmdGetSensorValues(void)
{
	hprintf_P(PSTR("\nVolume: %f%c\nSensor Temp: %fC\n"),displayValues.liters,displayValues.literUnitChar,displayValues.tempSemi/2.0);
	if (nParams>0)
		ExtraIgnored(nParams);
	return 0;	// always succeeds
}

/*!	\brief Clears Error Flags and Counters

	Resets watchdog and brownout reset counters and the
	error flags in the display.
	\return always 0
*/
static int8_t	CmdClearErrors(void)
{
	wdrCnt=0;
	boCnt=0;
	ResetDisplayFlags();
	hprintf_P(PSTR("Errors cleared.\n"));
	if (nParams>0)
		ExtraIgnored(nParams);
	return 0;	// always succeeds
}


/*!	\brief Prints Values of System Sensors...

	... and some other useful information.
*/
static int8_t	CmdStat(void)
{
	hprintf_P(PSTR("\nVolume: %f%c\nSensor Temp: %fC\n"),displayValues.liters,displayValues.literUnitChar,displayValues.tempSemi/2.0);
	hprintf_P(PSTR("Display Temp: %6.1fC\n"),TempFromADC());
	hprintf_P(PSTR("Distance: %4.1fmm\n"),distance*1000.0);
	if (config.spontaneousReportVolume==0.0)
	{
		hprintf_P(PSTR("No spontaneous volume reports\n"));
	} 
	else
	{
		hprintf_P(PSTR("Spontaneus volume reporting if difference>%f liters\nLast volume reported:%fL"),config.spontaneousReportVolume,lastReportedVolume);
	}
	hprintf_P(PSTR("Unreg: %4.2fV Min: %3.2fV Max: %3.2fV Ripple: %3.2fV\n"),UnregFromADC(ADCData.named.ADC_Unreg),
		UnregFromADC(ADCDataMin.named.ADC_Unreg),UnregFromADC(ADCDataMax.named.ADC_Unreg),
		UnregFromADC(ADCDataMax.named.ADC_Unreg-ADCDataMin.named.ADC_Unreg));
	hprintf_P(PSTR("Switched Unreg: %4.2fV\n"),SWUnregFromADC(ADCData.named.ADC_SWUnreg));
	hprintf_P(PSTR("5V Supply: %3.2fV Min: %3.2fV Max: %3.2fV Ripple: %3.2fV\n"),P5FromADC(ADCData.named.ADC_P5),
		P5FromADC(ADCDataMin.named.ADC_P5),P5FromADC(ADCDataMax.named.ADC_P5),
		P5FromADC(ADCDataMax.named.ADC_P5-ADCDataMin.named.ADC_P5));
	hprintf_P(PSTR("3V Supply: %3.2fV\n"),P3FromADC(ADCData.named.ADC_P3));
	hprintf_P(PSTR("Time: %2d:%02d:%02d\n"),hrCnt,minCnt,secCnt);
	hprintf_P(PSTR("DCF in Sync: %s\n"),DCFInSync?"Yes":"No");
	if (DCFInSync)
	{
		hprintf_P(PSTR("DCF difference (positive means clock is slow): %ds\n"),GetDCFdiff());
	}
	hprintf_P(PSTR("Clock adjust: %s\n"),RTCAdjTxt[RTCAdj]);
	hprintf_P(PSTR("Sounds are played from %d to %d o'clock\n"),config.sndHourFrom,config.sndHourTo);
	if (config.sndMask & SNDMASK_ALERT1ALWAYS)
	{
		hprintf_P(PSTR("Alert 1 sound is played always\n"));
	}
	if (config.sndMask & SNDMASK_ALERT2ALWAYS)
	{
		hprintf_P(PSTR("Alert 2 sound is played always\n"));
	}
	if (config.sndMask & SNDMASK_HOURBELL)
	{
		hprintf_P(PSTR("Big Ben is played every hour\n"));
	}
	if (BTStatus.haveBT)
	{
	hprintf_P(PSTR("Bluetooth detected. Remote Address: %02x:%02x:%02x:%02x:%02x:%02x Local port: %d\n"),
		BTStatus.remoteAddress[5],BTStatus.remoteAddress[4],BTStatus.remoteAddress[3],
		BTStatus.remoteAddress[2],BTStatus.remoteAddress[1],BTStatus.remoteAddress[0],
		BTStatus.localPort);
	hprintf_P(PSTR("Unknown Bluetooth telegrams: %d\n"),BTStatus.unknTelCnt);
	}
	hprintf_P(PSTR("Host commands lost: %d\n"),hostCommandsLost);
	hprintf_P(PSTR("Reset counters: Watchdog=%d  Brownout=%d\n"),wdrCnt,boCnt);
	if (displayValues.error!=NULL)
	{
		hprintf_P(PSTR("Display Error: %s\n"),displayValues.error);
	}
	if (channelMismatchCnt>0)
	{
		hprintf_P(PSTR("Channel Mismatch Count: %d\n"),channelMismatchCnt);
	}
if (nParams>0)
		ExtraIgnored(nParams);
	return 0;
}

/*!	\brief	Sets the Brightness of the LCD Backlight

If the brightness is a pure number, it is treated as an absolute value,
a preceding '+' or '-' increases or decreases the brightness for the
specified value.
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetBright(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetBright <x>\nx=0..255\n"));
		MissingArg();
		return -1;
	}
	int16_t	current=LCDGetBrightness();
	int16_t	param=atoi(params[0]);
	if ((params[0][0]=='+') || (params[0][0]=='-'))	// Relative value
	{
		current+=param;
	} 
	else
	{
		current=param;
	}
	if (current>255)
	{
		current=255;
	}
	if (current<0)
	{
		current=0;
	}
	LCDSetBrightness(current);
	config.LCDBright=current;
	configDirty=true;
	hprintf_P(PSTR("Brightness is %d\n"),LCDGetBrightness());
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Sets the Contrast of the LCD Backlight

If the contrast is a pure number, it is treated as an absolute value,
a preceding '+' or '-' increases or decreases the contrast for the
specified value.
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetContrast(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetContr <x>\nx=0..255\n"));
		MissingArg();
		return -1;
	}
	int16_t	current=LCDGetContrast();
	int16_t	param=atoi(params[0]);
	if ((params[0][0]=='+') || (params[0][0]=='-'))		// Relative value
	{
		current+=param;
	} 
	else
	{
		current=param;
	}
	if (current>255)
	{
		current=255;
	}
	if (current<0)
	{
		current=0;
	}
	LCDSetContrast(current);
	config.LCDBright=current;
	configDirty=true;
	hprintf_P(PSTR("Contrast is %d\n"),LCDGetContrast());
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Sets the Time
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetTime(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetTime hh:mm\n"));
		MissingArg();
		return -1;
	}
	int	hr,min;
	sscanf(params[0],"%d:%d",&hr,&min);
	if (hr<0 || hr>23 || min<0 || min>59)
	{
		hprintf_P(PSTR("Invalid time\n"));
		return -1;
	}
	cli();	// make this atomic
	hrCnt=hr;
	minCnt=min;
	secCnt=0;
	DCFInSync=false;
	sei();
	hprintf_P(PSTR("Time is %d:%02d %x %x\n"),hrCnt,minCnt);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Writes the Configuration to the EEPROM
	\return	0 on success, -1 on error.
*/
static int8_t	CmdWrite(void)
{
	if (configDirty)
	{
		WriteConfig();
		hprintf_P(PSTR("Configuration written.\n"));
	} else
	{
		hprintf_P(PSTR("Didn't write anything because configuration has not changed.\n"));
	}
	if (nParams>0)
	{
		ExtraIgnored(nParams);
	}
	return 0;
}

/*!	\brief	Sets the DCF-Pin logic
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetDCFInv(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: DCFInv <x>\nx=0 for normal, x=1 for inverted\n"));
		MissingArg();
		return -1;
	}
	uint8_t x=atoi(params[0]);
	if (x==0)
	{
		config.invDCF77=0;
	} 
	else
	{
		config.invDCF77=DCF77_PINMASK;
	}
configDirty=true;
hprintf_P(PSTR("DCF77-Pin is "));
hprintf_P((config.invDCF77==0? PSTR("normal\n"):PSTR("inverted\n")));
return 0;
}

/*!	\brief	Receives a Double from the Host
	\return	 The Value
*/
static double	EnterDouble(void)
{
	AssertRTS();
	while (!(cFlags&CFLAG_CMDRECEIVED))
	{
		while (!(adcFlags&ADCFLAG_NEWAVERAGES))
		{
			wdt_reset();
			sleep_enable();
			sleep_cpu();
			sleep_disable();
		}
		adcFlags&= ~ADCFLAG_NEWAVERAGES;
		if (config.displayType==DISPLAY_4x20)
		{
			LCDGoTo(0);
			printf_P(PSTR("P3:     %4d %6.2f"),ADCDataAvg.named.ADC_P3,P3FromADC(ADCDataAvg.named.ADC_P3));
			LCDGoTo(64);
			printf_P(PSTR("P5:     %4d %6.2f"),ADCDataAvg.named.ADC_P5,P5FromADC(ADCDataAvg.named.ADC_P5));
			LCDGoTo(20);
			printf_P(PSTR("Unreg:  %4d %6.2f"),ADCDataAvg.named.ADC_Unreg,UnregFromADC(ADCDataAvg.named.ADC_Unreg));
			LCDGoTo(84);
			printf_P(PSTR("SWUnreg:%4d %6.2f"),ADCDataAvg.named.ADC_SWUnreg,SWUnregFromADC(ADCDataAvg.named.ADC_SWUnreg));
		}
	}
	cFlags&= ~CFLAG_CMDRECEIVED;
	return atof((char*)cmdline);
}

/*!	\brief	Calibrates the Voltage Sensors

Normal operation is halted during the calibration so
the RTC will be late afterwards.<br>
The user is prompted to enter the voltages from which
the real Volts per LSB for each channel is calculated.
	\note	The JTAGEN-fuse has to be disabled for an accurate Value of P3
and no JTAG debugger must be connected else you would end up in heavy
miscalibration of the P3-value!
	\return	0 on success, -1 on error.
*/
static int8_t	CmdCalibrate(void)
{
	double	d;
	uint8_t	tccr0sav=TCCR0;
	TCCR0=0;	// Stop Timer 0
	LCDClear();
	LCDSetBrightness(0);
	hprintf_P(PSTR("*** Calibration Procedure ***\nHit return to skip a value\nPlease enter the real Value for 3V\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.P3VoltsPerLSB=d/ADCDataAvg.named.ADC_P3;
		configDirty=true;
		hprintf_P(PSTR("P3VoltsPerLSB=%f\n"),config.P3VoltsPerLSB);
	}
	hprintf_P(PSTR("Please enter the real Value for 5V\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.P5VoltsPerLSB=d/ADCDataAvg.named.ADC_P5;
		configDirty=true;
		hprintf_P(PSTR("P5VoltsPerLSB=%f\n"),config.P5VoltsPerLSB);
	}
	hprintf_P(PSTR("Please enter the real Value for Unreg\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.UnregVoltsPerLSB=d/ADCDataAvg.named.ADC_Unreg;
		configDirty=true;
		hprintf_P(PSTR("UNregVoltsPerLSB=%f\n"),config.UnregVoltsPerLSB);
	}
	SensorPowerOn();
	hprintf_P(PSTR("Please enter the real Value for Switched Unreg\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.SWUnregVoltsPerLSB=d/ADCDataAvg.named.ADC_SWUnreg;
		configDirty=true;
		hprintf_P(PSTR("SWUnregVoltsPerLSB=%f\n"),config.SWUnregVoltsPerLSB);
	}
//	sleep_disable();
	SensorPowerOff();
	TCCR0=tccr0sav;	// Restart Timer 0
	InitADC();	// Restore ADC configuration
	hprintf_P(PSTR("\nInstrument calibration.\nPress + or - to adjust for 100%, Space to accept, other key to skip.\n"));
	OCR1C=~config.instCal;
	cFlags&=~CFLAG_BYTERECEIVED;
	AssertRTS();
	for (;;)
	{
		wdt_reset();
		if (cFlags&CFLAG_BYTERECEIVED)
		{
			cFlags&=~CFLAG_BYTERECEIVED;
			if (byteFromHost=='-' && OCR1CL!=255)
			{
				OCR1CL++;
				hprintf_P(PSTR("%d\n"),OCR1CL);
			}
			else if (byteFromHost=='+' && OCR1CL!=0)
			{
				OCR1CL--;
				hprintf_P(PSTR("%d\n"),OCR1CL);
			}
			else if (byteFromHost==' ')
			{
				config.instCal=~OCR1CL;
				configDirty=true;
				hprintf_P(PSTR("New calibration value accepted.\n"));
				break;
			}
			else if (byteFromHost=='\n')
			{
				AssertRTS();
			}
			else
			{
				hprintf_P(PSTR("Instrument calibration aborted.\n"));
				break;
			}
		}
	}
	for (uint8_t i=0; i<=100;i+=20)
	{
		hprintf_P(PSTR("%d\n"),i);
		SetAnalogPerCent((double)i);
		delay10ms(255);	// Wait for 2.55 seconds
	}
	LCDClear();
	hprintf_P(PSTR("End calibration procedure.\n"));
	AssertRTS();
	usart1FlushRx();
	return 0;
}

#ifdef WITH_UPDATE
/*!	\brief	Updates the Sensor Firmware

The User is asked to send the Intel HEX file which is
then sent to the sensor.
	\return	0
*/
static int8_t	CmdUpdateSensorFirmware(void)
{
	UpdateSensorFirmware();
	if (nParams>0)
	{
		ExtraIgnored(nParams);
	}
	return 0;
}

/*!	\brief	Verifies the Sensor Firmware against an Intel-HEX-File

The User is asked to send the Intel HEX file which is
then verified against the sensor firmware.
	\return	0
*/
static int8_t	CmdVerifySensorFirmware(void)
{
	VerifySensorFirmware();
	if (nParams>0)
	{
		ExtraIgnored(nParams);
	}
	return 0;
}
#endif

/*!	\brief	Sets Debug Options

Possible parameters are <i>dcf</i> and <i>sensor</i>. Adding a parameter switches
debug output on, omitting it off. So debug without parameters completely
disables debugging output.
	\note	Be careful using debug output especially for DCF77. This includes calling
interrupt serviced routines (UART) from an interrupt and may lock up if the transmit
buffer is filled (e.g. by auxiliary communication) since in ATMega-architecture
interrupts and non-interruptible!
	\return 0
*/
static int8_t	CmdDebug(void)
{
	dFlags=0;
	for (uint8_t i=0;i<nParams;i++)
	{
		if (!strcmp_P(params[i],PSTR("sensor")))
		{
			dFlags|=DFLAG_SENSOR;
		} else
		if (!strcmp_P(params[i],PSTR("dcf")))
		{
			dFlags|=DFLAG_DCF77;
		} else
		if (!strcmp_P(params[i],PSTR("adc")))
		{
			dFlags|=DFLAG_ADC;
		} else
		{
			hprintf_P(PSTR("Unknown debug flag %s\nValid flags: adc dcf sensor\n"),params[i]);
		}
	}
	return 0;
}	

/*!	\brief	Sets the Thresholds for a Signaling Output
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetSignal(void)
{
	if (nParams<3)
	{
		hprintf_P(PSTR("Usage: SetSignal <number> <low> <high>\n"));
		MissingArg();
		return -1;
	}
	uint8_t number=atoi(params[0])-1;
	if (number<0 || number>1)
	{
		hprintf_P(PSTR("Sorry, we have only 2 signal outputs!\n"));
		return -1;
	}
	config.signalOutputs[number].activeBelow=atoi(params[1]);
	config.signalOutputs[number].activeAbove=atoi(params[2]);
	configDirty=true;
	if (config.signalOutputs[number].activeBelow>config.signalOutputs[number].activeAbove)
	{
		hprintf_P(PSTR("Warning: low>high, Output %d will be always on!\n"),number+1);
	}
	DumpConfigSignal(number);
	if (nParams>3)
	{
		ExtraIgnored(nParams-3);
	}
	return 0;
}

/*!	\brief	Sets the Sounds-OK-Time and Soundmask
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetSounds(void)
{
	if (nParams<2)
	{
		hprintf_P(PSTR("Usage: SetSounds <Hour from> <Hour to> [<Soundmask>]\n"));
		MissingArg();
		return -1;
	}
	uint8_t hourFrom=atoi(params[0]);
	uint8_t hourTo=atoi(params[1]);
	if ((hourFrom>24) || (hourTo>24))
	{
		hprintf_P(PSTR("Hour must be between 0 and 24!\n"));
		return -1;
	}
	configDirty=true;
	if (nParams>2)
	{
		config.sndMask=atoi(params[2]);
	}
	if (nParams>3)
	{
		ExtraIgnored(nParams-3);
	}
	return 0;
}

/*!	\brief	Sets the Delay for automatic Display stepping
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetPTime(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetPTime <seconds>\n"));
		MissingArg();
		return -1;
	}
	config.pageTime=atoi(params[0]);
	if (config.pageTime==0)
	{
		config.pageTime++;
	}
	configDirty=true;
	hprintf_P(PSTR("Display Page stepping time set to %d seconds\n"),config.pageTime);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Sets the Delay for switchback from manual to automatic Display stepping
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetPTimeK(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetPTimeK <seconds>\n"));
		MissingArg();
		return -1;
	}
	config.pageTimeK=atoi(params[0]);
	configDirty=true;
	if (config.pageTimeK==0)
	{
		config.pageTimeK++;
	}
	hprintf_P(PSTR("Switchback time from manual to automatic display page stepping set to %d seconds\n"),config.pageTime);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Causes a Watchdog Reset. Useful for Debugging

	\note The Display will show watchdog resets after this command.
		Use a clean power up or a "ClearErrors" to get rid of these.
	\return	This function does not return.
*/
static	int8_t	CmdReset(void)
{
	hprintf_P(PSTR("Bye.\n"));
	for(;;);
	return 0;	// Never reached
}
static int8_t	CmdHelp(void);

/*!	\brief	Performs a Factory Test
	\return	This function does not return. It ends with a device reset.
*/
static int8_t CmdFTest(void)
{
	
	hprintf_P(PSTR("Factory Test\n"));
	if (FTest())
	{
		hprintf_P(PSTR("*** Test PASSED ***\n"));
	} else
	{
		hprintf_P(PSTR("*** Test FAILED ***\n"));
	}
	return CmdReset();
}

/*!	\brief	Sets the Cistern Metrics
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetCistern(void)
{
	if (nParams<4)
	{
		hprintf_P(PSTR("Usage: SetCistern <const|horiz|sphere> <depth> <area|diameter> <capacity> [<length>]\n"));
		MissingArg();
		return -1;
	}
	if (!strcmp_P(params[0],PSTR("const")))
		config.cisternType=CIST_CONST_AREA;
	else if (!strcmp_P(params[0],PSTR("horiz")))
		config.cisternType=CIST_HORIZ_CYL;
	else if (!strcmp_P(params[0],PSTR("sphere")))
		config.cisternType=CIST_SPHERE;
	else
		hprintf_P(PSTR("Unknown cistern type: %s"),params[0]);
	config.cistHeight=atof(params[1]);
	config.cistAreaOrDiameter=atof(params[2]);
	config.cistCapacity=atof(params[3]);
	if (nParams>4)
	{
		config.cistLength=atof(params[4]);
	}
	configDirty=true;
	if (nParams>5)
	{
		ExtraIgnored(nParams-5);
	}
	return 0;
}

/*!	\brief	Sets the Display type
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetDisplay(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetDisplay <4x20|2x16|analog>\n"));
		MissingArg();
		return -1;
	}
	if (!strcmp_P(params[0],PSTR("analog")))
	{
		config.displayType=DISPLAY_ANALOG;
		configDirty=true;
	}
	else if (!strcmp_P(params[0],PSTR("2x16")))
	{
		config.displayType=DISPLAY_2x16;
		LCDClear();
		configDirty=true;
	} else if (!strcmp_P(params[0],PSTR("4x20")))
	{
		config.displayType=DISPLAY_4x20;
		configDirty=true;
	} else
	{
		hprintf_P(PSTR("Unknown display type %s. Display type not changed.\n"),params[0]);
	}
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Sets Spontaneous Volume Reporting
	\return	0 on success, -1 on error.
*/
static int8_t CmdSetSpontaneous(void)
{
	double	diff;
	
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetSpontaneous <liters>\n"));
		MissingArg();
		return -1;
	}
	diff=atof(params[0]);
	if (diff>=0)
	{
		config.spontaneousReportVolume=diff;
		configDirty=true;
	}
	else
	{
		hprintf_P(PSTR("Spontaneous volume is an absolute value and must not be negative.\nVolume not changed.\n"));
	}
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Set the analog instrument to a certain position

	This command is only useful for debugging
	if no other part of the software is setting the value!
*/
static int8_t	CmdAnalog(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: Analog <x>\nx=0..100 (in \%)\n"));
		MissingArg();
		return -1;
	}
	double	param=atof(params[0]);
	SetAnalogPerCent(param);
	hprintf_P(PSTR("OCR1CL=%d\n"),OCR1CL);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Sets the OCR1CL to an absolute Value

	This command is only useful for debugging
	if no other part of the software is setting the value!
	\return	Always 0
*/
static int8_t	CmdSetOcr(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: OCR <x>\nx=0..255\n"));
		MissingArg();
		return -1;
	}
	OCR1CL=atoi(params[0]);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

#ifdef USE_RTS
/*!	\brief	Sets if to use RTS/CTS-Handshake.
	\return	0 if OK, -1 on error
*/
static int8_t	CmdUseRts(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: UseRts <x>\nx=[0|1]\n"));
		MissingArg();
		return -1;
	}
	int8_t	useRts=atoi(params[0]);
	config.useRts=(useRts!=0);
	configDirty=true;
	return 0;
}
#endif

/*!	\brief	Continuously prints the Sensor Distance Value

	This command terminates if you press any key on the console.
	Mostly used for debugging and sensor installation.
	\note
	Normal operation is suspended while this command is running.
	\return	Always 0
*/
static int8_t	CmdDistance(void)
{
	SensorPowerOn();
	cFlags=0;
	AssertRTS();
	for (;;)
	{
		while (!(cFlags&CFLAG_TIMERECEIVED))
		{
			if (cFlags&CFLAG_BYTERECEIVED)
			{
				SensorPowerOff();
				return 0;
			}
		}			
		cFlags&= ~CFLAG_TIMERECEIVED;
		pulseTime=(double)pulseTimeMult/F_CPU*(double)pulseTimeInt;
		distance=speedOfSound*pulseTime/2.0;
		hprintf_P(PSTR("%d:%02d:%02d Time: %d Distance: %1.3fm %6.1f%c\n"),hrCnt,minCnt,secCnt,pulseTimeInt,distance,LitersFromDist(distance),displayValues.literUnitChar);
		wdt_reset();
		sleep_enable();
		sleep_cpu();
		sleep_disable();
	}
	return 0;
}

static const char helpCalibrate[]	PROGMEM	= "Calibrates the voltage sensors";
static const char helpClearErrors[]	PROGMEM	= "Clear error flags and counters";
static const char helpDCFinv[]		PROGMEM	= "Sets DCF77-pin polarity";
static const char helpDebug[]		PROGMEM	= "Change debug settings";
static const char helpDumpConf[]	PROGMEM	= "Prints a human readable list of configuration data";
static const char helpFTest[]		PROGMEM	= "Executes the factory test";
static const char helpGetSensorValues[]	PROGMEM = "Gets sensor measurement values";
static const char helpHelp[]		PROGMEM	= "This list of commands";
static const char helpReset[]		PROGMEM	= "Restarts the device";
static const char helpSetBright[]	PROGMEM	= "Sets backlight brightness, absolute or relative (e.g. 50, +10 -10)";
static const char helpSetCistern[]	PROGMEM	= "Sets the cistern metrics";
static const char helpSetContr[]	PROGMEM	= "Sets LCD contrast, absolute or relative (e.g. 25, +5 -5)";
static const char helpSetDisplay[]	PROGMEM	= "Sets the display type (4x20, 2x16 or analog)";
static const char helpSetPTime[]	PROGMEM	= "Sets the delay for automatic display stepping (0 means no automatic stepping)";
static const char helpSetPTimeK[]	PROGMEM	= "Sets the delay for switchback from manual to automatic stepping";
static const char helpSetSignal[]	PROGMEM	= "Sets the thresholds for signaling outputs";
static const char helpSetSounds[]	PROGMEM	= "Sets the Sounds-OK-Time and Soundmask. E.g. SetSounds 9 18 0";
static const char helpSetSpontaneous[]	PROGMEM = "Sets Volume difference for spontaneous reporting (0=Off)";
static const char helpSetTime[]		PROGMEM	= "Sets the time. E.g. SetTime 18:35";
#ifdef USE_RTS
static const char helpUseRts[]		PROGMEM = "Use RTS/CTS-handshake for serial communication";
#endif
static const char helpStatus[]		PROGMEM	= "Prints current status";
static const char helpDistance[]	PROGMEM	= "Continuously prints sensor distance value until you press a key";
static const char helpUpdate[]		PROGMEM	= "Updates sensor firmware";
static const char helpVerify[]		PROGMEM	= "Verifies sensor firmware";
static const char helpWrite[]		PROGMEM	= "Writes configuration to non-volatile memory.\n\n*** Execute Write after any changes in configuration\n or your changes will get lost at next reset! ***\n\n";

static const commandTable_t	commandTable[]= {
	{"Calibrate",CmdCalibrate,helpCalibrate},
	{"ClearErrors",CmdClearErrors,helpClearErrors},
	{"DCFInv",CmdSetDCFInv,helpDCFinv},
	{"Debug",CmdDebug,helpDebug},
	{"Distance",CmdDistance,helpDistance},
	{"DumpConf",CmdDumpConf,helpDumpConf},
	{"FTest",CmdFTest,helpFTest},
	{"GetSensorValues",CmdGetSensorValues,helpGetSensorValues},
	{"Help",CmdHelp,helpHelp},
	{"Reset",CmdReset,helpReset},
	{"SetBright",CmdSetBright,helpSetBright},
	{"SetCistType",CmdSetCistern,helpSetCistern},
	{"SetContr",CmdSetContrast,helpSetContr},
	{"SetDisplay",CmdSetDisplay,helpSetDisplay},
	{"SetPTime",CmdSetPTime,helpSetPTime},
	{"SetPTimeK",CmdSetPTimeK,helpSetPTimeK},
	{"SetSignal",CmdSetSignal,helpSetSignal},
	{"SetSounds",CmdSetSounds,helpSetSounds},
	{"SetSpontaneous",CmdSetSpontaneous,helpSetSpontaneous},
	{"SetTime",CmdSetTime,helpSetTime},
	{"Status",CmdStat,helpStatus},
#ifdef USE_RTS
	{"UseRts",CmdUseRts,helpUseRts},
#endif
#ifdef WITH_UPDATE
	{"Update",CmdUpdateSensorFirmware,helpUpdate},
	{"Verify",CmdVerifySensorFirmware,helpVerify},
#endif
	{"Write",CmdWrite,helpWrite},
	/*	The following are debug-commands that are not officially documented.
		Since they have NULL help texts, they are not listed in the help-command.
	*/
	{"Analog",CmdAnalog,NULL},	// Sets instrument value in percent
	{"OCR",CmdSetOcr,NULL},	// Directly sets the instrument's OCR
};

/*!	\brief Lists a Description of all Commands
*/
static int8_t	CmdHelp(void)
{
	hprintf_P(PSTR("\nValid commands:\n"));
	for (uint8_t i=0;i<sizeof(commandTable)/sizeof(commandTable_t);i++)
	{
		if (!commandTable[i].helptext)	// no help text, it is a secret command
		{
			continue;
		}
		hprintf_P(PSTR("%16.16s "),commandTable[i].cmd);
		hprintf_P(commandTable[i].helptext);
		hprintf_P(PSTR("\n"));
	}
	if (nParams>0)
		ExtraIgnored(nParams);
	return 0;
}


/*!	\brief Executes one command line
	\param str A pointer to the line
	\return 0 on success, 1 on error
*/
uint8_t	ExecCommand(char *str)
{
	cmd=str;
	while (*cmd==' ')
		cmd++;	// skip leading spaces
	if (*cmd=='\0')
	{
		return 1;	// There is no command at all!
	}
	char *cp;
	nParams=0;
	cp=cmd;
	do
	{
		while ((*cp!=' ') && (*cp))
			cp++;	// Advance to next space or end
		if (*cp==' ')
			{
				*cp='\0';	// Terminate the previous param
				cp++;
			}
		while (*cp==' ')	// Skip spaces
			cp++;
		if (*cp)	// Next parameter discovered
		{
			if (*cp=='"')	// It is a String
			{
				cp++;
				char *cp2=strchr(cp,'"');
				if (cp2)	// Hope the string ends somewhere...
				{
					*cp2=0;	// Terminate it
				}
				params[nParams]=cp;
				cp=cp2+1;
			}
			else
			{
				params[nParams]=cp;
			}
			nParams++;
		}
	} while (*cp);
	uint8_t cmdlen=strlen(cmd);
	// Now execute the command
	for (uint8_t i=0;i<sizeof(commandTable)/sizeof(commandTable_t);i++)
	{
		if (strncasecmp(commandTable[i].cmd,cmd,cmdlen)==0)
		{
			commandTable[i].adr();
			if (configDirty)
			{
				hprintf_P(PSTR("* Don't forget to WRITE the configuration to the EEPROM!\n"));
			}
			return 0;
		}
	}
	hprintf_P(PSTR("Error: unknown command \"%s\"\n"),cmd);
	return 1;
}
