/*! \file stackchk.c
	\brief	Contains Routines to check maximum Stack Usage

	\copyright	Copyright (C) 2015 Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
	
	This module tries to estimate stack usage. This is done by filling the
	memory from __bss_end to RAMEND with a predefined value. Calls of
	Stackchk() find the first bytes that differs and fills the stackchk
	structure.<br>
	Note that this function need to check <b>every</b> single byte from
	__bss_end to the first difference and thus might take some milliseconds
	to complete. It should not be called from ISRs.<br>
	This is not a secure method since your application might push exactly
	this byte for an unlimited number of times  without a chance to detect it
	but under normal conditions it would work.
	You can call this function regularly or just before you need the values
	and they will you give an idea how large your stack can grow.
	If you have only a few bytes reserve or there might be functions
	using large amount of stack space that have not been called yet,
	you should think about a better solution...<br>
*/ 

#include <avr/io.h>
#include <stdlib.h>
#include "stackchk.h"

extern char __bss_end;	///< The lowest SRAM-Address not used for any Variable
StackchkStruct	stackchk __attribute__ ((section (".noinit")));

/*!	\brief	Do not call this function from anywhere! It is called automatically
			by the startup Routine.
*/
static void __attribute__ ((naked, used, section(".init3"))) StackchkFill(void)
{
#ifdef WITH_STACKCHK
	char	*cp;
	
	for (cp=(char*)&__bss_end;cp <(char*)RAMEND;cp++)
		{
			*cp=STACKCHKFILLBYTE;
		}
	stackchk.stackmin=(uint16_t)RAMEND;
	stackchk.heapmax=(uint16_t)&__bss_end;
#endif
}

/*!	\brief	Fills the stackchk Struct

	It walks through the free RAM to find the probable end of the
	heap and the beginning of the stack.
	Note that there is no guarantee that heap nor stack cannot grow
	more than this but it can give you an idea of how many spare bytes
	you have before heap and stack collide.<br>
	Heap end is detected if four consecutive fill bytes are found.
	In extreme cases this may happen inside the heap as well.
	Also, the last bytes on the stack may accidentally be the same than
	the fill byte and the real stack size would not be correctly found.
	You might want to use different fill bytes to get more exact results.
	You must decide yourself ho probable the fill byte is in your
	application.<br>
	You can calculate the reserve before heap and stack would collide as
	<pre>stackchk.stackmin-stackchk.heapmax-1</pre>
*/
void	Stackchk(void)
{
	#ifdef WITH_STACKCHK
	char*	cp;
	uint8_t	cnt;
	
	cp=(char*)&__bss_end;	// The first free Byte
	cnt=0;
	do 
	{
		if (*cp==STACKCHKFILLBYTE)
			{
				cnt++;
				if (cnt==4)	// Found four consecutive fill bytes, assume untouched area begins
				{
					cp-=3;
					break;
				}
			}
		else
			{
				cnt=0;
			}
		cp++;
		if (cp==(char*)RAMEND)	// Did not find fill bytes in the whole area! Stack is probably corrupted!
		{
			stackchk.heapmax=(uint16_t)&__bss_end;	// Don't know where heap ends and stack begins!
			stackchk.stackmin=stackchk.heapmax+1;
			return;
		}
	} while (1);
	if (stackchk.heapmax<(uint16_t)cp-1)
	{
		stackchk.heapmax=(uint16_t)cp-1;
	}
	else
	{
		cp=(char*)stackchk.heapmax+1;	// Start from the highest location found until now
	}
	for (;cp<(char*)RAMEND;cp++)
	{
		if (*cp!=STACKCHKFILLBYTE)
			break;
	}
	if (stackchk.stackmin>(uint16_t)cp)
	{
		stackchk.stackmin=(uint16_t)cp;
	}
	return;
#endif
}
