﻿/*! \file ihex.c
	\brief	Contains Routines to decode Intel HEX Files and to send them to the Sensor

	\copyright	Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

	Since the boot loader timing is very strict (there must not be a pause of more
	than a second between two characters) and we have enough of RAM we decode the
	Intel Hex file into RAM and then start the programming sequence.
*/ 


#include <stdint.h>
#include <stdlib.h>
#include <avr/wdt.h>
#include <avr/pgmspace.h>
#include "display.h"
#include "uart.h"
#include "delay.h"
#include "ihex.h"

#define MEMSIZE	0x800	///< Maximum size of the Sensor Program
#define MAXBYTES	30	///< Maximum number of Bytes per Line (including Address, Type and Byte Count)

#ifdef WITH_UPDATE
/*!	mem[] is big enough to hold the whole program memory of the sensor.
	This takes a lot of RAM but we have enough to maximize reliability.
	We can load the whole bunch to RAM first and securely transfer it to the sensor.	*/
static char	mem[MEMSIZE];
/*!	\brief Parses one Line of an Intel HEX File
	\return	-1 on Error, 0 on Success, 1 on End of File
*/
static int8_t	DoOneLine(void)
{
	uint8_t	i;
	char	*cmdPtr;
	char	hex[3];
	uint8_t	bytes[MAXBYTES];
	wdt_reset();
	hex[2]=0;	// String termination
	cmdPtr=(char*)&cmdline[1];
	for (i=0;i<MAXBYTES;i++)
	{
		hex[0]=*cmdPtr++;
		hex[1]=*cmdPtr++;
		bytes[i]=(uint8_t)strtoul(hex,0,16);
	}
	uint8_t	type=bytes[3];
	uint16_t	adr=bytes[1]<<8|bytes[2];
	uint8_t	len=bytes[0];
	if (type==RECTYPE_DATA)
	{
		if (adr+len>MEMSIZE)
		{
			hprintf_P(PSTR("Error: Address > MEMSIZE\n"));
			return -1;
		}
		for (i=0;i<len;i++)
		{
			mem[adr++]=bytes[i+4];
		}
	} 
	else if (type==RECTYPE_EXTSEG)
	{
		if (bytes[4]|bytes[5])	// Segment address !=0
		{
			hprintf_P(PSTR("Error: Segment Record is not supported\n"));
			return -1;
		}
	} else if (type==RECTYPE_EOF)
	{
		return 1;
	}
	return 0;
}

/*!	\brief	Reads an Intel HEX File into Memory
	\return	0 on Success, -1 on Error
*/
static int8_t	ReadIntelHex(void)
{
	int8_t	cc;
	
	for (uint16_t i=0;i<MEMSIZE;i++)	// Erase memory buffer
	{
		mem[i]=0xff;
	}
	for (;;)
	{
		AssertRTS();
		while (!(cFlags&CFLAG_CMDRECEIVED))
			{
				wdt_reset();
			}
		cFlags&= ~CFLAG_CMDRECEIVED;
		if (cmdline[0]!=':')	// Does not look like a Intel HEX file
		{
			hprintf_P(PSTR("Error: line does not start with \":\"\n"));
			return -1;
		}
		cc=DoOneLine();
		if (cc<0)
		{
			hprintf_P(PSTR("Error processing line:\n%s\n"),cmdline);
			return -1;
		} else if (cc==1)
		{
			hprintf_P(PSTR("Intel HEX file sucessfully read\n"));
			return 0;
		}
	}
}

/*!	\brief	Prints one line of Hexdump
	\param	adr	The Address to be displayed
	\param	ptr	A Pointer to the first Byte
	\param	n The number of Bytes to dump
*/
void	HexDump(uint16_t adr,void *ptr, uint8_t n)
{
	hprintf_P(PSTR("%04x"),adr);
	for (uint8_t i=0;i<n;i++)
	{
		hprintf_P(PSTR(" %02x"),*(char*)ptr++);
		wdt_reset();
	}
	hprintf_P(PSTR("\n"));
}

/*!	\brief	Waits one Second
	\note This function uses the hard-timed delay50ms and
		therefore does not rely on interrupts being enabled.
*/
void	Delay1s()
{
	for (uint8_t i=0;i<20;i++)
	{
		delay50ms();
		wdt_reset();
	}
}


/*!	\brief	Reads one page of Sensor ROM via the Sensor Boot Loader

The page is stored in bootLdPage[]
	\param	adr The start Address
	\return	-1 on Error, 0 on Success
*/
int8_t BootldRead(uint16_t adr)
{
	blFlags=0;	// Reset Boot Loader Flags
	putc0('?');
	putd0((char)(adr>>8));
	putd0(adr&0xff);
	putd0(cksum0);
	putc0(0);	// end of telegram
	wdt_reset();	// If nothing comes back from the sensor, we will have a watchdog reset here.
	while(!(blFlags&(BOOTLDDATA|BOOTLDERROR))); // wait for confirmation
	if (blFlags&BOOTLDERROR)
	{
		return -1;
	} else
	{
		return 0;
	}
}

/*!	\brief	Sends one Page to the Sensor Boot Loader
	\param	adr The Start Address of the Page
	\return	-1 on Error, 0 on Success
*/
int8_t BootldWrite(uint16_t adr)
{
	uint8_t i;

	blFlags=0;	// Reset Boot Loader Flags
	putc0('!');
	putd0((uint8_t)(adr>>8));
	putd0((uint8_t)(adr&0xff));
	for(i=0;i<32;i++) {
		putd0(mem[adr++]);
	}
	putd0(cksum0);
	putc0(0);	// end of telegram
	while(!blFlags) // wait for confirmation
	{
		i=tickCnt+20;
		if (i>=100)
		{
			i-=100;
		}
		while (tickCnt!=i);
//		hprintf_P(PSTR("%02x "),blFlags);
//		wdt_reset();
	}
	if (blFlags&BOOTLDERROR)
	{
		return -1;
	} else
	{
		return 0;
	}
}

/*!	\brief	Programs the Sensor Firmware with the Content of mem[]
	\note	This function requires the Boot Loader activated.
\return	0 if OK, -1 on Error
*/
static int	ProgramSensor(void)
{
	hprintf_P(PSTR("Programming...\n"));
	for (int16_t i=0;i<MEMSIZE;i+=32)
	{
		wdt_reset();
		hprintf_P(PSTR("#%04x "),i);
		int8_t	cc=BootldWrite(i);
		hprintf_P(PSTR("cc:%d blFlags:%02x\n"),cc,blFlags);
		if(cc<0)
		{
			hprintf_P(PSTR("\nBoot loader error.\n"));
			return cc;
		}
	}
	return 0;
}

/*!	\brief	Verifies the Sensor Firmware against the Content of mem[]
	\note	This function requires the Boot Loader activated.
\return	0 if OK, -1 on Error
*/
static int8_t VerifySensor(void)
{
	int8_t	err=0;
	
	hprintf_P(PSTR("\nVerifying...\n"));
	for (int16_t i=0;i<MEMSIZE;i+=32)
	{
		wdt_reset();
		hprintf_P(PSTR("."));
		int8_t cc=BootldRead(i);
		if(cc<0)
		{
			hprintf_P(PSTR("Boot loader error.\n"));
			return -1;
		}
		for (int8_t	j=0;j<32;j++)
		{
			if (bootLdPage[j]!=mem[i+j])
			{
				hprintf_P(PSTR("\nVerify error at %04x.\nSensor: "),i+j,*(char*)0x3f);
				HexDump(i,bootLdPage,32);
				hprintf_P(PSTR("File:   "));
				HexDump(i,&mem[i],32);
				err=1;
				break;
			}
		}
	}
	if (!err)
		hprintf_P(PSTR("\nVerify OK.\n"));
	return 0;
}

/*!	\brief	Updates the Sensor Firmware
	\return	0 on Success, -1 on Error
*/
int8_t	UpdateSensorFirmware(void)
{
	int8_t	cc;
	
	SensorPowerOff();	// Reset sensor
	hprintf_P(PSTR("Waiting for Intel HEX file...\n"));
	cc=ReadIntelHex();
	if (cc<0)
	{
		return cc;
	}
	SensorPowerOn();
	blFlags=0;
	while(!(blFlags&BOOTLDHELLO)) wdt_reset();	// Waiting for boot loader. Watchdog will reset if it does not talk to us.
	hprintf_P(PSTR("Bootloader activated.\n"));
//	delay5ms();
	putc0('#');	// activate boot loader
	ProgramSensor();
	return VerifySensor();
}

/*!	\brief	Verifies the Sensor Firmware against an Intel-Hex-File
	\return	0 on Success, -1 on Error
*/
int8_t	VerifySensorFirmware(void)
{
	int8_t	cc;
	
	SensorPowerOff();	// Reset sensor
	hprintf_P(PSTR("Waiting for Intel HEX file...\n"));
	cc=ReadIntelHex();
	if (cc<0)
	{
		return cc;
	}
	SensorPowerOn();
	blFlags=0;
	while(!(blFlags&BOOTLDHELLO)) wdt_reset();	// Waiting for boot loader. Watchdog will reset if it does not talk to us.
	hprintf_P(PSTR("Bootloader activated.\n"));
//	delay5ms();
	putc0('#');	// activate boot loader
	return VerifySensor();
}

#endif