/*! \file ftest.c
	\brief	Contains Factory Test Routines

	\copyright	Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/ 

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include "delay.h"
#include "display.h"
#include "ftest.h"
#include "uart.h"
#include "key.h"
#include "adc.h"
#include "led.h"
#include "sensors.h"

/*!	\brief Ports to test

	You must specify every port in your system in ascending order (even if it has not
	to be tested) and every	bit that is set will be included in the test.<br>
	Only pins configured as output can be tested!
*/
const static ftestPortsStruct	ports[]=
{
	{
		&PORTA,&PINA,0b11111111
	},
	{
		&PORTB,&PINB,0b11111111
	},
	{
		&PORTC,&PINC,0b11111111
	},
	{
		&PORTD,&PIND,0b11010011
	},
	{
		&PORTE,&PINE,0b11111100
	},
	{
		&PORTF,&PINF,0b00001111
	},
	{
		&PORTG,&PING,0b00011111
	},
};

static  uint8_t port;	///< The Port currently under test
static  uint8_t bit;	///< The Pin of port currently under test. This is a bit mask.
static  uint8_t testPort;	///< The Port tested against current port
static  uint8_t	tpport;	///< The Read-Value of testPort PORT value
static  uint8_t	tppin;	///< The Read-Value of testPort PIN value
static  uint8_t	tptestMask;	///< The Test Mask of testPort

/*!	\brief	Converts a Byte into a binary String
	\param	byte The byte
	\param	str	A String capable of holding at least the 9 characters representing the binary value
	\return	A pointer to a string
	\note	This function is <b>not reentrant</b>!
		The string <b>must</b> be used before the next call
		since the address is static!
		Multiple arguments in a single printf <b>will not work</b>!
		Split them into several printfs and <b>do not</b> call this function
		from ISRs!
*/
static const char *Byte2bin(uint8_t byte,char str[9])
{
	uint8_t	i=0;
	for (uint8_t mask=128; mask!=0; mask>>=1)
	{
		str[i]=(byte&mask)?'1':'0';
		i++;
	}
	str[8]=0;	// Terminating zero
	return str;
}

/*!	\brief	Prints an Error Message for conflicting Pins
*/
static void	PrintError()
{
	char	str[9];
hprintf_P(PSTR("port %c, pin %d, testport %c, readport %s, "),'A'+port,bit,'A'+testPort,Byte2bin(tpport,str));
hprintf_P(PSTR("readpin %s, "),Byte2bin(tppin,str));
hprintf_P(PSTR("tptestmask %s, "),Byte2bin(tptestMask,str));
hprintf_P(PSTR("eor %s\n"),Byte2bin(tpport ^ tppin,str));
}

/*!	\brief	Prints PASS or FAIL
	\param	pass True if passed
*/
static void	PrintPass(bool pass)
{
	if (pass)
	{
		hprintf_P(PSTR("PASS\n"));
	} 
	else
	{
		hprintf_P(PSTR("FAIL\n"));
	}
}

/*!	\brief	Waits for the Key to be pressed and released

	The LCD-backlight is blinking until the key is pressed.
*/
static void	WaitKeyBlinking(void)
{
	kflags=0;
	while (!(kflags&KFLAG_KEYRELEASED))
	{
		wdt_reset();
		sleep_enable();
		sleep_cpu();
		sleep_disable();
		if (tickCnt<50)	// Blink the LCD backlight
		{
			LCDSetBrightness(255);
		} else
		{
			LCDSetBrightness(0);
		}
	}
}

/*!	\brief Tests the Switched Unreg Node
	\return	true if test passed, false if not
*/
static inline bool	FTestSWUnreg()
{
	double	VSWUnreg,threshold,time;
	bool	testPassed=true;

	SensorPowerOff();	// Switch sensor off
	delay1s();	// Ensure that sensor power is discharged
	threshold=UnregFromADC(ADCDataAvg.named.ADC_Unreg)*0.9;
	VSWUnreg=SWUnregFromADC(ADCDataAvg.named.ADC_SWUnreg);
	if (VSWUnreg>1.0)	// Still more than 1V? There must be something wrong!
	{
		testPassed=false;
		hprintf_P(PSTR("SWUnreg should be 0 but is %5.2fV\n"),VSWUnreg);
	}

	hprintf_P(PSTR("Switching SWUnreg ON - "));
	TCNT0=0;
	tickCnt=0;
	SensorPowerOn();
	for(;;)
	{
		adcFlags=0;
		while (!(adcFlags&ADCFLAG_NEWRESULTS));	// Wait for next measurement
		if (SWUnregFromADC(ADCData.named.ADC_SWUnreg)>=threshold)
		{
			cli();
			time=(((uint16_t)tickCnt<<8)|TCNT0)*10.0/256.0;
			sei();
			hprintf_P(PSTR("Rise time to 90%% (%5.2fV): %6.2fms\n"),threshold,time);
			break;
		}
		if (tickCnt==99)
		{
			testPassed=false;
			hprintf_P(PSTR("Timeout waiting for SWUnreg to rise. Either I cannot measure it or it does not rise. Please check.\n"));
			break;
		}
		wdt_reset();
	}
	while (tickCnt!=99)	// Allow SWUnreg to settle
	{
		wdt_reset();
	}
	threshold=UnregFromADC(ADCDataAvg.named.ADC_Unreg)*0.1;

	hprintf_P(PSTR("Switching SWUnreg OFF - "));
	TCNT0=0;
	tickCnt=0;
	SensorPowerOff();
	for(;;)
	{
		adcFlags=0;
		while (!(adcFlags&ADCFLAG_NEWRESULTS));	// Wait for next measurement
		if (SWUnregFromADC(ADCData.named.ADC_SWUnreg)<=threshold)
		{
			cli();
			time=(((uint16_t)tickCnt<<8)|TCNT0)*10.0/256.0;
			sei();
			hprintf_P(PSTR("Fall time to 10%% (%5.2fV): %6.2fms.\n"),threshold,time);
			break;
		}
		if (tickCnt==99)
		{
			testPassed=false;
			hprintf_P(PSTR("Timeout waiting for SWUnreg to fall.\n"));
			break;
		}
		wdt_reset();
	}
	PrintPass(testPassed);
	return (testPassed);
}

/*!	\brief	Short Circuit Test #1

All pins included in the testMask[] are set to 1.
A single "0" is shiftet through all pins, the pin-registers
are read to verify its presence exactly on this position.
	\return	True on test pass, False on fail
*/
static inline bool	FTestWalkingZero()
{
	static uint8_t	nPorts=sizeof(ports)/sizeof(ftestPortsStruct);
	bool	testPassed=true;
	
	hprintf_P(PSTR("Walking zero\n"));
	for (uint8_t i=0;i<nPorts;i++)	// Set all pins to 1
	{
		*ports[i].port |= ports[i].testMask;
	}
	for (port=0;port<nPorts;port++)
	{
		for (bit=1;bit!=0;bit<<=1)
		{
			if (!(ports[port].testMask & bit))
				continue;
			*ports[port].port=0xff & ports[port].testMask & ~bit;	// Clear one bit
			delay50us();	// Settling time. The test is fast enough even with this.
			for (testPort=0;testPort<nPorts;testPort++)	// Not check weather other ports remain unchanged
			{
				tpport=*ports[testPort].port;
				tppin=*ports[testPort].pin;
				tptestMask=ports[testPort].testMask;
				if ((tpport ^ tppin) & tptestMask)
				{
					PrintError();
					testPassed=false;
					wdt_reset();
				}
			}
			*ports[port].port=0xff & ports[port].testMask;	// Set the bit again
		}
	}
	PrintPass(testPassed);
	return testPassed;
}

/*!	\brief	Short Circuit Test #2

All pins included in the testMask[] are set to 0.
A single "1" is shiftet through all pins, the pin-registers
are read to verify its presence exactly on this position.
	\return	True on test pass, False on fail
*/
static inline bool	FTestWalkingOne(void)
{
	static uint8_t	nPorts=sizeof(ports)/sizeof(ftestPortsStruct);
	bool	testPassed=true;

	hprintf_P(PSTR("Walking one\n"));
	for (uint8_t i=0;i<nPorts;i++)	// Set all pins to 0
	{
		*ports[i].port &= ~ports[i].testMask;
	}
	for (port=0;port<nPorts;port++)
	{
		for (bit=1;bit!=0;bit<<=1)
		{
			if (!(ports[port].testMask & bit))
			continue;
			*ports[port].port |= bit;	// Set one bit
			delay50us();	// Settling time
			for (testPort=0;testPort<nPorts;testPort++)	// Not check weather other ports remain unchanged
			{
				tpport=*ports[testPort].port;
				tppin=*ports[testPort].pin;
				tptestMask=ports[testPort].testMask;
				if ((*ports[testPort].port ^ *ports[testPort].pin) & ports[testPort].testMask)
				{
					PrintError();
					testPassed=false;
					wdt_reset();
				}
			}
			*ports[port].port &= ~bit;	// Set the bit again
		}
	}
	PrintPass(testPassed);
	return testPassed;
}

/*!	\brief	Toggles the Optocoupler Outputs

	This is not an automated test. The user has to verify
	that LEDs connected to the optocouplers will toggle.
*/
static inline void	FTestToggleSignal()
{	// Flicker the signal outputs. LEDs connected to the optocouplers will give feedback to the user.
	hprintf_P(PSTR("Toggling Sig1\n"));
	for (uint8_t i=0;i<10;i++)
	{
		Sig1ON();
		delay50ms();
		Sig1OFF();
		delay50ms();
		wdt_reset();
	}
	hprintf_P(PSTR("Toggling Sig2\n"));
	for (uint8_t i=0;i<10;i++)
	{
		Sig2ON();
		delay50ms();
		Sig2OFF();
		delay50ms();
		wdt_reset();
	}
}

/*!	\brief	Test Supply Voltages
	\note	Again the advice for P3: the pin measuring P3 is shared with the
	JTAG-interface. If JTAG is enabled by its fuse, P3-value will be inaccurate
	by up to 15% FSR, if a JTAG debugger is connected it is completely unmeasurable!
	Disable the JTAGEN-fuse or this test <b>will</b> fail!
	\return	True on test pass, False on fail
*/
static inline bool	FTestSupply(void)
{
	bool	testPassed=true;
	double v;
	
	v=UnregFromADC(ADCDataAvg.named.ADC_Unreg);
	hprintf_P(PSTR("Testing Supply Voltages\nUnreg: %5.2fV\n"),v);
	hprintf_P(PSTR("Unreg: %5.2fV (%.1f%%)\n"),v,(v/15.0-1.0)*100.0);
	if (v>15.0)
	{
		PrintPass(true);
	} else
	{
		testPassed=false;
		PrintPass(false);
	}
	v=P5FromADC(ADCDataAvg.named.ADC_P5);
	hprintf_P(PSTR("5.0V: %5.2fV (%.1f%%)\n"),v,(v/5.0-1.0)*100.0);
	if ((v>5.0*0.95) && (v<5.0*1.05))
	{
		PrintPass(true);
	} else
	{
		testPassed=false;
		PrintPass(false);
	}
	v=P3FromADC(ADCDataAvg.named.ADC_P3);
	hprintf_P(PSTR("3.3V: %5.2fV (%.1f%%)\n"),v,(v/3.3-1.0)*100.0);
	if ((v>3.3*0.95) && (v<3.3*1.05))
	{
		PrintPass(true);
	} else
	{
		if (v<.1)
		{
			hprintf_P(PSTR("WARNING: 3V3-supply is almost zero. Maybe this device does not have "
			"a bluetooth module. If so, you can safely ignore this error\n"));
		}
		testPassed=false;
		PrintPass(false);
	}
	return testPassed;
}

/*!	\brief	Performs a simple Factory Test

	This routine tries to test everything that can be tested without
	user intervention. It cannot detect all possible error conditions
	but will detect a lot of possible short circuits and other errors
	and can give valuable hints to the test personal.<br>
	All special function pins must be set to normal port I/O.
	The testMask basically is a copy of the DDR since only
	output pins can be tested. Take care of pins that must
	keep their special function during the test (e.g. TxD of
	the USART that receives the result). This has also to be masked
	out, as well as pins that would cause disastrous consequences
	if switched uncontrolled.<br>
	First the user is asked to press the pushbutton key. This is to test
	the key itself. After that, the signal outputs are toggled a few times
	each. LEDs connected on the test adapter will show the user if
	that works. The following tests require no user attention.
	The device will measure its supply voltages and
	test all output pins against all others so short circuits between
	output pins or output pin and VCC/GND will most likely be detected.<br>
	Since interrupts are disabled in the progress of this function and
	DDRs are modified, the calling routine should perform a watchdog
	reset to restore proper operation of the device.<br>
	Do not try to restore DDRs and interrupts, you will most likely
	forget some and there is nothing bad in willingly performing a reset...
	\return True if no error detected, False if test fails
*/
bool	FTest(void)
{
	bool	testPassed=true;
	
	// First a few Tests we need the running System for.
	hprintf_P(PSTR("Press pushbutton key to start the test.\nSystem will reboot afterwards.\n"));
	SensorPowerOff();
	WaitKeyBlinking();				// This is the test for the key
	FTestToggleSignal();	// Signal outputs have to be checked visually by the user
	if (!FTestSupply())
	{
		testPassed=false;
	}
	if (!FTestSWUnreg())	// Test the SWUnreg current source
	{
		testPassed=false;
	}
	// And now the "destructive" Test. Reduce system Activity to Factory Test Level.
	TIMSK=0;				// Disable timer 0 interrupt
	ADCSRA&= 0b11110111;	// Disable ADC interrupt
	// reset special function pins to normal operation
	TCCR3A&=0b00000011;
	TCCR1A&=0b00000011;
	DDRF=0x0f;	// Override voltage sense resistors for testing
	DDRA=0xff;	// DFC77 pin is set to output for testability. No receiver must be connected!
	DDRB=0xff;	// Key signal is set to output. Do not press the key during test.
	LEDGreenOff();
	LEDBlueOff();
	LEDYellowOff();
	LEDRedOff();
	delay1s();
	LEDRedOn();
	delay1s();
	LEDGreenOn();
	delay1s();
	LEDYellowOn();
	delay1s();
	LEDBlueOn();
	delay1s();
	if (!FTestWalkingZero())
	{
		testPassed=false;
	}
	if (!FTestWalkingOne())
	{
		testPassed=false;
	}
	return testPassed;
}

/*!	\brief	Get RxD0-Pin Value
	\return	True if RxD0=1
*/
bool GetRxD0(void)
{
	return (PINE&1)!=0;
}

/*!	\brief	Get RxD1-Pin Value
	\return	True if RxD1=1
*/
bool GetRxD1(void)
{
	return (PIND&4)!=0;
}

/*!	\brief	Sets the state of TxD0
	\param	level	The desired Output Level
*/
void	SetTxD0(bool level)
{
	if (level)
	{
		PORTE |=2;
	} else
	{
		PORTE &= ~2;
	}
}

/*!	\brief	Sets the state of TxD1
	\param	level	The desired Output Level
*/
void	SetTxD1(bool level)
{
	if (level)
	{
		PORTD |=8;
	} else
	{
		PORTD &= ~8;
	}
}

/*!	\brief	Returns Connections between the two RS485 Interfaces
	\return	0 if they are not connected<br>
		1 if only TxD0 is connected to RxD1<br>
		2 if only TxD1 is connected to RxD0<br>
		3 if they are fully connected
*/
uint8_t FTest485GetConnects(void)
{
	uint8_t	connects=0;
	bool	i;
	
	Uart1DisableLineDriver();
	Uart0EnableLineDriver();
	delay50us();
	i=GetRxD1();
	SetTxD0(!i);
	delay50us();	// The line drivers are relatively slow
	if (i!=GetRxD1())
	{
		connects+=1;
	}
	Uart0DisableLineDriver();
	Uart1EnableLineDriver();
	delay50us();
	i=GetRxD0();
	SetTxD1(!i);
	delay50us();
	if (i!=GetRxD0())
	{
		connects+=2;
	}
	return connects;
}

/*!	\brief	Performs a Factory Test of the RS485-Lines

	For this test to succeed, all jumpers on X3 have to be
	in position "485" and a switch has to be in the test
	adapter that connects sensor RS485 with host RS485.
*/
void FTest485(void)
{
	uint8_t	connects;
	UCSR0B=0;	// Disable the USARTs
	UCSR1B=0;
	connects=FTest485GetConnects();
	if (connects>0)
	{
		LCDPrintLine_P(0,PSTR("Open Loop Switch"));
		LCDPrintLine_P(1,PSTR("connects: %d"),connects);
		while ((connects=FTest485GetConnects())>0)
		{
			if (secCnt>15)	// Was 0 when this function has been called. We don't wait longer than 15s
			{
				LCDPrintLine_P(0,PSTR("Timeout"));
				for(;;);	// Watchdog reset
			}
			wdt_reset();
		}
	}
	LCDPrintLine_P(0,PSTR("Open Test PASSED"));
	LCDPrintLine_P(1,PSTR("Close Loop Switch"));
	while (FTest485GetConnects()==0)	// Wait for the connection to appear
	{
		if (secCnt>15)	// Was 0 when this function has been called. We don't wait longer than 15s
		{
			LCDPrintLine_P(0,PSTR("Timeout"));
			for(;;);	// Watchdog reset
		}
		wdt_reset();
	}
	delay10ms(50);	// Wait for the switch to be stable
	connects=FTest485GetConnects();
	if (connects!=3)
	{
		LCDPrintLine_P(0,PSTR("Loop Test FAILED"));
		LCDPrintLine_P(1,PSTR("connects: %d"),connects);
	} else
	{
		LCDPrintLine_P(0,PSTR("Loop Test PASSED"));
		LCDPrintLine_P(1,PSTR("Bye."));
	}
	delay10ms(255);
	for(;;);
}