﻿/*! \file dcf77.c

	\brief	Routines for setting and adjusting the Real Time Clock via DCF77
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */ 

#include <stdlib.h>
#include <avr/pgmspace.h>
#include "uart.h"
#include "dcf77.h"
#include "config.h"
#include "led.h"

bool	DCFInSync;	///< True when a valid Telegram has been received.
int32_t	DCFdiff;	///< Difference to RTC in Seconds, positive means RTC is late.
	
static const uint8_t	ms100min=6;	///< The minimum Pulse Width to be recognized as 100ms
static const uint8_t	ms100max=14;	///< The maximum Pulse Width to be recognized as 100ms
static const uint8_t	ms200min=15;	///< The minimum Pulse Width to be recognized as 200ms
static const uint8_t	ms200max=23;	///< The maximum Pulse Width to be recognized as 200ms
static const uint8_t	mspause59min=150;	///< The minimum Pulse Width to be recognized as a missing pulse
static const uint8_t	mspause59max=220;	///< The maximum Pulse Width to be recognized as a missing pulse
static bool	minpulseReceived;	///< Did we receive a valid Minute Pulse (missing Pulse at second 59)?
static bool	lastPinState;	///< The Pin State at last Interrupt
static const uint8_t	xorMask=0;	///< set to 0 or DCF77_PINMASK to invert signal
static uint8_t	dcfTicks;	///< counts 10ms-intervals between the edges of the signal
static uint8_t	bitcnt;		///< counts the bits in the telegram
uint8_t	min;	///< The Minute Value received
uint8_t	hour;	///< The Hour Value received

/*!	\brief Gets the difference between DCF and RTC in seconds.
	Positive means RTC is late.
*/
int32_t	GetDCFdiff()
{
	return DCFdiff;
}

/*!	\brief Returns the logical Signal of the DCF Pin.

	This function honors the invDCF77 configuration variable
	and returns a polarity corrected value of the DCF77 receive signal.
	\return 0 for pause, 1 for pulse
*/
inline static uint8_t GetPinState(void)
{
	if ((DCF77_PIN & DCF77_PINMASK) ^ config.invDCF77)
	{
		LEDBlueOn();
		return 1;
	} else
	{
		LEDBlueOff();
		return 0;
	}
}


/*!	\brief Invalidates Reception Variables, so Reception is started from scratch
*/
void Invalidate()
{
	minpulseReceived=false;
	DCFInSync=false;
//	lastPinState=0;
	dcfTicks=0;
	bitcnt=0;
	min=0;
	hour=0;
	RTCAdj=RTC_NORMAL;
}

/*!	\brief	Stores a Bit in the next Position of the Telegram
*/
static void PushBit(uint8_t b)
{
	if (bitcnt>=36)	// Bits above 36 are date bits. We don't care about them.
	{
	} else
	if (bitcnt>=29)	// here comes an hour-bit
	{
		hour>>=1;
		if (b)
		{
			hour|=64;
		}
	} else
	if (bitcnt>=21)	// here comes a minute-bit
	{
		min>>=1;
		if (b)
		{
			min|=128;
		}
	} else
	if (bitcnt==20)
	{
		if (!b)	// bit 20 has to be "1"
		{
			if (dFlags&DFLAG_DCF77)
			{
				hprintf_P(PSTR("DCF77: bit20 not 1\n"));
			}
			Invalidate();
		}		
	} else
	if (bitcnt==0)
	{
		if(b)	// bit 0 has to be "0"
		{
			if (dFlags&DFLAG_DCF77)
			{
				hprintf_P(PSTR("DCF77: bit0 not 0\n"));
			}
			Invalidate();
		}
	}
	bitcnt++;
}

/*!	\brief	Counts the Number of 1-Bits in a Byte
	\param	b The Byte
	\return	 The Number of Bits set
*/
static uint8_t	BitsSet(uint8_t b)
{
	uint8_t pop = 0;
    while (b) {
        pop++;
        b = b & (b - 1);
    }
	return pop;
}

/*!	\brief	Interrupt Handler for DCF77

Has to be called from 10ms interrupt
*/
void DCF77int(void)
{
	bool	pinState;
	
	dcfTicks++;
	if (dcfTicks==0)	// overflow
	{
		Invalidate();
	}
	pinState=GetPinState();
	if (pinState!=lastPinState)
	{
		lastPinState=pinState;
		if (pinState==0)	// this is the end of a pulse
		{
			if (dcfTicks>=ms100min && dcfTicks <=ms100max)
			{
				PushBit(0);
			} else if (dcfTicks>=ms200min && dcfTicks <=ms200max)
			{
				PushBit(1);
			} else
			{
				Invalidate();
			}
		} else	// this is the end of a pause
		{
			if (dcfTicks>=mspause59min && dcfTicks <=mspause59max)
			{
				if (minpulseReceived)	// we have received a complete telegram
				{
					if ((BitsSet(hour)&1)==0 && (BitsSet(min)&1)==0)	// parity ok?
					{
						hour=((hour>>4)&0x03)*10+(hour&0x0f);
						min=((min>>4)&0x07)*10+(min&0x0f);
						if (hour<24 && min<60)	// final plausibility check
						{
							if (dFlags&DFLAG_DCF77)
							{
								hprintf_P(PSTR("DCF: %2d:%02d\n"),hour,min);
							}
							RTCsync();
						}
					} else
					{
						Invalidate();
						if (dFlags&DFLAG_DCF77)
						{
							hprintf_P(PSTR("Parity hour:%d:%d min:%d:%d\n"),hour,BitsSet(hour),min,BitsSet(min));
						}							
					}
				}
				minpulseReceived=true;
				bitcnt=0;
				min=0;
				hour=0;
			}
		}
		dcfTicks=0;
	}
}