﻿/*! \file config.h

	\brief	Structures and Constants for Configuration Handling
	
	\copyright Copyright (C) 2013  Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef CONFIG_H_
#define CONFIG_H_

#include <stdbool.h>
#include <stdio.h>
#include "Display.h"

#define CONF_MAGIC	0x3797	///< Change this value if confStruct_t becomes incompatible to previous versions!
#define NUM_SIGNAL	2	///< Number of available signal outputs

#define SNDMASK_ALERT1ALWAYS	1	///< Signal 2 Alert is played outside silent period, too
#define SNDMASK_ALERT2ALWAYS	2	///< Signal 1 Alert is played outside silent period, too
#define SNDMASK_HOURBELL		4	///< Big Ben is played every hour

//! Holds the Thresholds for a Signal Optocoupler Output
typedef struct  
{
	uint16_t	activeBelow;	///< Threshold below this the output is active
	uint16_t	activeAbove;	///< Threshold above this the output is active
} signalOutput_t;

typedef enum {
	CIST_CONST_AREA,	///< Cistern with constant area over volume
	CIST_HORIZ_CYL,		///< Cistern of horizontal cylinder type
	CIST_SPHERE			///< Spherical cistern
	} cisternType_T;

//! Holds all Configuration Data stored in EEPROM
typedef struct
{	uint16_t		magic;	//!< Used to discover unprogrammed Devices
	displayType_t	displayType;	//!< The display type used
	cisternType_T	cisternType;	//!< The shape of the cistern
	double	cistAreaOrDiameter;	//!< Surface Area of the Cistern (const) or Diameter (horizontal Cylinder or Sphere)
	double	cistLength;	//!< Length of the Cistern (horizontal Cylinder Type only)
	double	cistHeight;	//!< Height of the Sensor over Cistern Ground
	double	cistCapacity;	//!< Capacity of the cistern in liters (used for %-calculation)
	uint8_t	invDCF77;	//!< Invert the DCF77 Pin?
	uint8_t	LCDV0;	//!< Control Voltage for LCD Contrast
	uint8_t	LCDBright;	//!< LCD Brightness
	double	P3VoltsPerLSB;	//!< Calibration Value for 3V3
	double	P5VoltsPerLSB;	//!< Calibration Value for 5V
	double	UnregVoltsPerLSB;	//!< Calibration Value for Unreg
	double	SWUnregVoltsPerLSB;	//!< Calibration Value for Switched Unreg
	bool	backlightAlwaysOn;	///< Shall the Backlight always be lit?
	uint8_t	backlightOnTime;	///< Time the Backlight will stay on after Key Press in Seconds
	uint8_t	pageTime;	///< Delay for automatic stepping of display pages (0 means no automatic stepping)
	uint8_t	pageTimeK;	///< Delay for switchback from manual stepping by key to automatic stepping
	signalOutput_t	signalOutputs[NUM_SIGNAL];	///< Thresholds for the signalling Outputs
	uint8_t	sndHourFrom;	///< Sounds are only played if current hour is greater or equal
	uint8_t	sndHourTo;	///< Sounds are only played if current hour is less than this
	uint8_t	sndMask;	///< Flags for special sound treatment
	uint8_t	instCal;	///< Calibration value for analog instrument
	bool	useRts;	///< Shall the serial communication use RTS/CTS-handshake?
	double	spontaneousReportVolume;	///< Volume is sent over serial if difference to last reported is greater
} confStruct_t;
extern bool	configDirty;

extern confStruct_t	config;
void	ReadConfig(void);
void	WriteConfig(void);
void	DumpConfigSignal(uint8_t number);
void	DumpConfig(void);
bool	SoundsOK(void);

#endif /* CONFIG_H_ */