/*! \file scale.c

	\brief	The Main Program and miscellaneous Routines
	
	\copyright Copyright (C) 2018 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
*/


#include <avr/io.h>
#include <avr/lock.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <stdlib.h>
#include <stdbool.h>
#include "scale.h"
#include "config.h"
#include "display.h"
#include "bin2bcd.h"

/*!	\brief	Fuse settings for the Production ELF file
*/
#if defined (__AVR_ATmega8__) || defined (__AVR_ATmega8A__)
#elif defined (__AVR_ATmega48PA__)
FUSES =
{
	.low = (FUSE_SUT_CKSEL1 & FUSE_CKOUT),	// Internal RC oscillator
	.high = (FUSE_BODLEVEL0 & FUSE_BODLEVEL1 & FUSE_SPIEN & FUSE_EESAVE),
	.extended = (FUSE_SELFPRGEN),
};
#else
#error Unknown Controller Type
#endif

LOCKBITS = (LB_MODE_1);	///< The Lock Bits for the production File

volatile uint8_t	tFlags;		///< Flags set by Time
volatile uint8_t	eFlags;		///< Error flags
volatile uint8_t	tickCnt;	///< Counter for timer0 interrupts
volatile uint8_t	secCnt;		///< Seconds of inactivity
volatile uint8_t	clickCnt;	///< Time since last click. Used to detect doubleclicks
static volatile delay_t	delaycnt;	///< Counter for delay(). Count value is milliseconds
volatile uint8_t	zeroWeightSecCnt;	///< Counts seconds of 'no load'. Used for accelerated power off
uint16_t	Temp;	///< Raw ADC Value of Chip Temperature. Typ. 354=85C, 292=25C, 225=-45C, ~1C/bit
uint16_t	VBat;	///< Raw ADC Value of Battery Voltage. 28.5mV/LSB or 29.19V FSR with a  120k/4k7 divider
volatile SWITCH_ACTION_t	switchAction;	///< Is set by interrupt if the switch has been pressed. Must be reset by the surrounding loop.
volatile weight_t weightRaw;	///< The raw Weight as received from the HX711
weight_t weightAvg;	///< The moving mean value of the raw weight values
weight_t weightZero;	///< The averaged weight at zero load

#if defined(__DOXYGEN__) || defined(USE_GetAvg1)
uint8_t histSize;
weight_t weightHist[MAXHIST];	///< the last 8 values used for averaging
uint8_t	weightIndex;	///< pointer to the next value of weightHist
weight_t weightSum;	///< the sum of all weightHist
#elif defined (__DOXYGEN__) || defined(USE_GetAvg2)
uint8_t avgFactor;
#elif defined (__DOXYGEN__) || defined(USE_GetAvg3)
// Different filter coefficients for testing. Uncomment the two lines for the one to use
// Attention: make sure NUM_COEFF in scale.h is set to the number of coefficients in coeff[]!

// Tom Roelands, https://fiiir.com/
//const int8_t coeff[NUM_COEFF]={0,-4,-8,20,88,127,88,20,-8,-4,0};	// filter coefficients
//const int16_t sumCoeff=319;	// sum of the coefficients, for normalization

// T-Filter, http://t-filter.engineerjs.com/
// 0Hz-0.5Hz: 1; 2Hz-5Hz: -50dB
//const int8_t coeff[NUM_COEFF]={6,21,47,78,105,115,105,78,47,21,6};	// filter coefficients
//const int16_t sumCoeff=629;	// sum of the coefficients, for normalization

// Tom Roelands, https://fiiir.com/
// fL=0.1, bL=3, Hamming
const int8_t coeff[NUM_COEFF]={10,21,50,86,116,127,116,86,50,21,10};	// filter coefficients
const int16_t sumCoeff=693;	// sum of the coefficients, for normalization

int32_t	weightHist[NUM_COEFF];
#endif

weight_t	lastWeightAvg;	///< The weightAvg 1 second ago (used to detect activity)
uint8_t	hxCnt;	///< is incremented each time the HX receives a new value
bool HXdisplayWeight;	///< If true, reading a new HX value updates the display. May be disabled for service display.

weight_t	sum128;	///< Contains the sum of 128 values when cnt128==128
uint8_t	cnt128;	///< Must be set to zero to start summing

/*!	\brief	List of possible reference Weights for Calibration
*/
const weight_t refWeights[]={
	100000,
	200000,
	500000,
	999999,
};


/*!	\brief	Reset Timeout Counters
*/
void	ResetTimeout(void)
{
	secCnt=0;
	zeroWeightSecCnt=0;
}

/*!	\brief	Gets the calibrated Weight from a raw HX-Value
	\param	raw The raw Value
	\return The calculated Weight according to Calibration Data
*/
weight_t	GetWeightFromRaw(int32_t raw)
{
	int32_t	weight;
	
	weight=(int64_t)(raw - weightZero) * config.cal_num / config.cal_denom;
	return (int32_t)	weight;
}

/*!	\brief Initializes the Weight Averaging
	
	Must be called once during startup
*/
static inline void	InitAVG(void)
{
#if defined(__DOXYGEN__) || defined(USE_GetAvg1)
	for (uint8_t i=0;i<=histSize;i++)
	{
		weightHist[i]=0;
	}
	weightSum=0;
#endif
}

#if defined(__DOXYGEN__) || defined(USE_GetAvg1)
/*!	\brief	Get averaged Weight

	This routine uses sliding averages i.e. the result
	is the average over the last n values.
	\param w The actual weight
	\return The average over the last HIST_SIZE values
*/
weight_t	GetAVG1(weight_t w)
{
	weightSum-=weightHist[weightIndex];	// subtract the oldest value
	weightHist[weightIndex]=w;	// store the new one
	weightSum+=w;
	weightIndex++;
	if (weightIndex>(histSize-1))
	{
		weightIndex=0;
	}
	return weightSum/histSize;
}
#endif

#if defined(__DOXYGEN__) || defined(USE_GetAvg2)
/*!	\brief	Get filtered Weight

	This routine uses an IIR filter to smoothen the display value.
	The formula is x<sub>i</sub>=(x<sub>(i-1)</sub> * c + x) / (c+1)
	\param w The actual weight
	\return The filtered weight over the last 8 values
*/
weight_t	GetAVG2(weight_t w)
{
	static int32_t	lastWeight;
	int32_t		weight;

	weight=(lastWeight*(avgFactor) + (int32_t)w) / (avgFactor+1);
	lastWeight=weight;
	return weight;
}
#endif

#if defined(__DOXYGEN__) || defined(USE_GetAvg3)
/*!	\brief	Get filtered Weight

	This routine uses an FIR SINC filter to smoothen the display value.
	\param w The actual weight
	\return The filtered weight over the last NUM_COEFF values
*/
weight_t	GetAVG3(weight_t w)
{
	static int8_t	histPointer,histIndex;
	int32_t		weight;

	histPointer++;
	if (histPointer>=NUM_COEFF)
	{
		histPointer=0;
	}
	weightHist[histPointer]=w;
	weight=0;
	histIndex=histPointer-1;
	for (uint8_t i=0;i<NUM_COEFF;i++)
	{
		if (histIndex<0)
		{
			histIndex=NUM_COEFF-1;
		}
		weight += weightHist[histIndex]*coeff[i];
		histIndex--;
	}
	weight/=sumCoeff;
	return weight;
}
#endif

/*!	\brief Reads out the HX711
	\param extraPulses
	the number of clock pulses after the
	value has been received.
	This selects the gain for the next measurement.
	may be 1 (g=128), 2 (g=32) or 3 (g=64)
	
	When a new Value is available (DOUT=0)
	it data is shifted out and weightRaw is set accordingly.
*/
void	readHX(uint8_t extraPulses)
{
	static weight_t	avg2;	// For averaging the averages to reduce display rate and flicker
	static uint8_t		avg2cnt=0;	// Counter for averaging
	
	if ((PIND & 0x4)==0)	// new value is ready
	{
		weightRaw=0;
		for (uint8_t i=24;i>0;i--)
		{
			cli();	// HX datasheet specifies max. 50us for the high time so we can not tolerate interrupts here
			PORTD |= 0x08;	// HX clock high
			weightRaw<<=1;	// the shift is placed here to give the pulse a longer high time
			PORTD &= ~0x08;	// HX clock low
			sei();	// high time is over, interrupts are ok from now.
			if (PIND & 0x04)
			{
				weightRaw|=1;
			}
		}
		while (extraPulses)
		{
			PORTD |= 0x08;	// HX clock high
			asm("NOP");
			asm("NOP");
			PORTD &= ~0x08;	// HX clock low
			extraPulses--;
		}
		if (weightRaw&0x00800000l)	// expand 24 bit signed int to 32 bit
		{
			weightRaw|=0xff000000l;
		}
//		weightRaw=testWeight;	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		if (cnt128!=128)	// summing of 128 values, used for calibration only
		{
			if (cnt128==0)
			{
				sum128=weightRaw;	// first value, init avg128
			} 
			else
			{
				sum128+=weightRaw;	// continue summing
			}
			cnt128++;
		}
		
		avg2cnt++;
#if defined USE_GetAvg1
		weightAvg=GetAVG1(weightRaw);
#elif defined USE_GetAvg2
		weightAvg=GetAVG2(weightRaw);
#elif defined USE_GetAvg3
		weightAvg=GetAVG3(weightRaw);
#endif
		avg2+=weightAvg;
		if (hxCnt!=0xff)	// counter shall not overflow
		{
			hxCnt++;
		}
		if (avg2cnt>3)
		{
			weight_t weight=GetWeightFromRaw(avg2/4);
			avg2=0;
			avg2cnt=0;
			if (HXdisplayWeight)
			{
				if (weight>=0)
				{
					Int24toDisplayHex(Bin6toBCD8(weight),&display[0]);
					if (display[0]!=chars[DISPLAY_SPACE])
					{
						display[0]|=0x80;	// set the decimal points
					}
					display[3]|=0x80;
				} 
				else
				{
					Int24toDisplayHex(Bin6toBCD8(-weight),&display[1]);
					display[0]=chars[DISPLAY_MINUS];
					if (display[1]!=chars[DISPLAY_SPACE])
					{
						display[1]|=0x80;
					}
					display[4]|=0x80;
				}
			}
		}
		
	}
}

/*!	\brief Handles the Switch

	This routine is called 100 times per second from
	the timer interrupt.
	If a switch event occurs it sets the switchAction
	variable accordingly.
	The surrounding loop must set it to SWITCH_NONE
	when it processes the event.
*/
static inline void readSwitch(void)
{
	static int16_t	pressedTime;
	
	if (PIND & 0x10)	// switch is pressed
	{
		if (pressedTime<0xffff)
		{
			pressedTime++;
			if (pressedTime==200)	// at 2s, reset weight. Do not wait for the switch to be released
			{
				switchAction=SWITCH_MED;
			}
			if (pressedTime==1000)	// at 10s, set display to CAL
			{
				SetDisplayCAL();
			}
		}
		displayBrightness=3;
		ResetTimeout();
	} 
	else // switch is released
	{
		if (pressedTime>5)	// switch has just been released, 5ms is used to debounce
		{
			ResetTimeout();	// reset time of inactivity
			switchAction=SWITCH_SHORT;
			if (clickCnt<50)	// last click is not more than 500ms ago -> doubleclick!
			{
				switchAction=SWITCH_DOUBLE;
			}
			clickCnt=0;
			if (pressedTime>200)
			{
				switchAction=SWITCH_MED;
				if (pressedTime>1000)
				{
					switchAction=SWITCH_LONG;
				}
			}
		}
		pressedTime=0;
	}
}

/*!	\brief	Timer compare interrupt (1000Hz)
*/
#if defined (__DOXYGEN__)
void TIMER_COMPA_vect(void)
#elif defined (__AVR_ATmega8__) || defined (__AVR_ATmega8A__)
ISR(TIMER1_COMPA_vect)
#elif defined (__AVR_ATmega48PA__)
ISR(TIMER0_COMPA_vect)
#endif
{
	static uint8_t	cnt10=1;	// counts from 10 to 1 to give 100Hz for some tasks

	if (delaycnt)	//  for delayms()
	{
		delaycnt--;
	}
	cnt10--;
	NextDigit();
	switch (cnt10)
	{
	case 0:	// Every 10ms (100Hz)
		cnt10=10;
		tickCnt++;
		if (tickCnt==100) {	// One second has passed
			tickCnt=0;
			tFlags|=TFLAG_SEC;
			if (secCnt!=255)
			{
				secCnt++;
			}
			if (((secCnt&0x03)==3) && (displayMode==DISPLAY_ALT_SHOW_NORMAL))	// show alternate display every fourth second
			{
				displayMode=DISPLAY_ALT_SHOW_ALT;
			}
			else
			{
				if (displayMode==DISPLAY_ALT_SHOW_ALT)
				{
					displayMode=DISPLAY_ALT_SHOW_NORMAL;
				}
			}
			if (secCnt<15)
			{
				displayBrightness=3;
			}
			switch (secCnt)
			{
				case 15:
					displayBrightness=2;
					break;
				case 30:
					displayBrightness=1;
					break;
				case 240:
				case 255:
					PowerOff();
					break;
			}
		}
		break;
	case 1:
		readSwitch();
		break;
	case 2:
		if (clickCnt<255)
		{
			clickCnt++;
		}
		if ((PINB & 0x08)==0)	// Jumper for always on
		{
			ResetTimeout();
		}
	}
}

/*!	\brief	Delays for ms Milliseconds

	\note The timer resolution is 1ms and the state of the
		timer when calling this function is normally undefined.
		The actual delay may differ up to 1ms from the given
		value (especially, it may be nearly 0 if you specify 1ms)!
	\param	ms The desired delay in Milliseconds
*/
void	DelayMs(delay_t ms)
{
	delaycnt=ms;
//	delaycnt=1;	// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	while (delaycnt!=0)
	{
		readHX(1);
		wdt_reset();
		sleep_cpu();
	}
}

/*!	\brief	Initializes the System
*/
void	Init(void)
{
	DDRB= 0b11110111;	// all lines output (even MISO to prevent a floating pin)
	PORTB=0b11001111;	// enable pullup on MOSI (to detect Power Save Disable)
	DDRC= 0b11011111;	// only MEAS-pin is input
	PORTC=0b00010000;	// SEG_D high
	DDRD= 0b11101011;	// DAT and SW are input
	PORTD=0b11010000;	// set power still off, must be set to on after a short delay, enable pullup on SW
#if defined (__AVR_ATmega8__) || defined (__AVR_ATmega8A__)
//	TCCR1A=(1<<WGM11) | (1<<WGM10);
	TCCR1B=(1<<CS11) | (1<<CS10) | (1<<WGM12);
	OCR1AL=124;
	TIMSK=(1<<OCIE1A);
#elif defined (__AVR_ATmega48PA__)
	TCCR0A=(1<<WGM01);	// CTC-mode
	TCCR0B=(1<<CS01)+(1<<CS00);	// Set timer 0 to CTC-mode, clock as Clk/64. f=125kHz
	OCR0A=124;			// gives 1000Hz
	TIMSK0|=(1<<OCIE0A);	// enable timer0 Output Compare Match Interrupt
	PRR=(1<<PRTWI) | (1<<PRTIM1) | (1<<PRTIM2) | (1<<PRSPI) | (1<<PRUSART0);
#endif
	DelayMs(50);
	PowerOn();
	wdt_enable(WDTO_1S);
	sleep_enable();
	DisplayNormal();
	displayBrightness=3;
#if defined USE_GetAvg1
	histSize=MINHIST;
	InitAVG();
#elif defined USE_GetAvg2
	avgFactor=MIN_AVG_WEIGHT;
#endif
	sei();
	ReadConfig();
}

/*!	\brief Writes VBat to display_alt

	Displays VBat on the rightmost 3 digits.
	Resolution is 100mV. VBat is only allowed up to 40V so
	3 digits are enough.
	"BAT" can be displayed at the leftmost digits
	if you remove the comments but it is intended that
	the left three digits show the temperature then.
	Anyway, this routine is intended for debugging
	purposes only.
	Display is set to alternate mode.
	Leading zero is suppressed what makes it more readable
	with common battery voltages.
*/
void	VBat2Display_alt()
{
	uint32_t	v,l;
	
	// Formula: Volts=VRef/1024*VBat/Rlow*(Rhigh+Rlow)
	// = 2.56/1024*VBat/4.7*124.7 = VBat*0.06633 or VBat/15.076
	// Any 10 bit value can be multiplied by up to 2^22 to fit into a uint32.
	// We want 100mV resolution so we calculate VBat*0.6633*2^22
	l=VBat*2782073;
	// Shift 22 bit to right and we have the digits
	l>>=22;
	v=Bin4toBCD5((uint16_t)l);
//	display_alt[0]=chars[0x0b];
//	display_alt[1]=chars[0x0a];
//	display_alt[2]=chars[DISPLAY_t];
	display_alt[3]=chars[(v>>8) & 0x0f];
	if (display_alt[3]==chars[0])	// suppress leading zero
	{
		display_alt[3]=chars[DISPLAY_SPACE];
	}
	display_alt[4]=chars[(v>>4) & 0x0f] | 0x80;	// Set the decimal point here
	display_alt[5]=chars[v & 0x0f];
}

/*!	\brief Handles Error Flags
*/
void	doEFlags(void)
{
	if (eFlags&EFLAG_UNCAL)
	{
		SetDisplayAltUncal();
	}
	if (eFlags&EFLAG_LOBAT)
	{
		SetDisplayAltLoBat();
	}
}
/*!	\brief Handles Timer Flags

	For now, there is only one flag that is set once per second.
	It is used to acquire the analog values for chip temperature
	and battery voltage and to detect weight changes for display
	recovery.
	Timer flags have to be cleared at the end of this routine.
*/
void	doTflags(void)
{
//	Int24toDisplay(weightAvg,&display[0]);

	ADMUX=(1<<REFS0)+(1<<REFS1)+8; // select temperature sensor
	ADCSRA|=(1<<ADEN)+(1<<ADSC);	// enable ADC, start conversion
	while (ADCSRA & (1<<ADSC));	// wait for conversion ready
//	Temp=(ADCH<<8)|ADCL;
	Temp=ADC;
	ADCSRA|=(1<<ADIF);	// clear interrupt flag
//	Int8toDisplayHex(Temp,&display_alt[0]);

	ADMUX=(1<<REFS0)+(1<<REFS1)+5; // select channel 5 (VBat)
	ADCSRA|=(1<<ADEN)+(1<<ADSC)+(1<<ADPS2)+(1<<ADPS1)+(1<<ADPS0);	// enable ADC, start conversion
	while (ADCSRA & (1<<ADSC));	// wait for conversion ready
	VBat=ADC;
	if (VBat<(uint16_t)(8.0/0.06633))
	{
		eFlags|=EFLAG_LOBAT;
	} else
	{
		eFlags&= ~EFLAG_LOBAT;
	}
	ADCSRA|=(1<<ADIF);	// clear interrupt flag
//	Int16toDisplay(VBat,&display_alt[2]);
//	VBat2Display_alt();
	if (abs(weightAvg-lastWeightAvg)>500)
	{
		ResetTimeout();
		displayBrightness=3;
	}
	lastWeightAvg=weightAvg;
	tFlags=0;
	// handle accellerated power off under no load condition
	if (abs(GetWeightFromRaw(weightAvg))<500)	// near zero (less than 5g)
	{
		zeroWeightSecCnt++;
		if (zeroWeightSecCnt>=15)
		{
			for (;;)
			{
				PowerOff();
			}
		}
	} 
	else
	{
		zeroWeightSecCnt=0;
	}
}

/*!	\brief Handles Switch Events
*/
void	doSwitch(void)
{
	switch (switchAction)
	{
		case SWITCH_NONE:	// This case is just to satisfy the compiler. It will not generate any code.
			break;	// this should not happen.
		case SWITCH_SHORT:	// Reactivate the display
			ResetTimeout();
			displayBrightness=3;
			break;
		case SWITCH_MED:	// Set the current weight to zero
			weightZero=weightAvg;
			break;
		case SWITCH_LONG:	// Enter calibration
			Calibrate();
			break;
		case SWITCH_DOUBLE:	// Doubleclick
#if defined USE_GetAvg1
			if (histSize==MINHIST)
			{
				histSize=MAXHIST;
				SetDisplayHiRes();
			} 
			else
			{
				histSize=MINHIST;
				SetDisplayLoRes();
			}
			InitAVG();	// must be reset else the sums will be wrong!
#elif defined USE_GetAvg2
			if (avgFactor==MIN_AVG_WEIGHT)
			{
				avgFactor=MAX_AVG_WEIGHT;
				SetDisplayHiRes();
			} 
			else
			{
				avgFactor=MIN_AVG_WEIGHT;
				SetDisplayLoRes();
			}
			// here, no reset is necessary!
#endif
			break;
	}
	switchAction=SWITCH_NONE;
}

/*!	\brief Performs Calibration of the Scale
*/
void	Calibrate(void)
{
	weight_t	zero,ref;
	int8_t	refIndex;
	
	HXdisplayWeight=false;
	DisplayNormal();
	// Let the user select a reference weight
	ResetTimeout();
	displayBlinkMask=BLINK_SLOW;
	while (true)
	{
		for (refIndex=0;refIndex<sizeof(refWeights)/sizeof(weight_t);refIndex++)
		{
			Int24toDisplayHex(Bin6toBCD8(refWeights[refIndex]),&display[0]);
			display[0]|=0x80;	// set the decimal points
			display[3]|=0x80;
			ResetTimeout();
			switchAction=SWITCH_NONE;
			while (switchAction!=SWITCH_SHORT)
			{
				wdt_reset();
				sleep_cpu();
				if (secCnt==5)
				{
					goto beginCalibrate;
				}
			}
		}
	}
	beginCalibrate:
	displayBlinkMask=BLINK_FAST;
	DelayMs(3000);
	displayBlinkMask=0;
	SetDisplayDontTouch();
	DelayMs(2000);
	cnt128=0;
	SetDisplayCLR(display_alt);
	DisplayAlt();
	while(cnt128!=128)
	{
		Int8toDisplayHex(127-cnt128,&display_alt[4]);
		zeroWeightSecCnt=0;
		doTflags();
		readHX(1);
		wdt_reset();
	}
	DisplayNormal();
	zero=sum128;
	SetDisplayRef();
	switchAction=SWITCH_NONE;
	while (switchAction!=SWITCH_SHORT)
	{
		zeroWeightSecCnt=0;
		doTflags();
		readHX(1);
		wdt_reset();
	}
	SetDisplayDontTouch();
	DelayMs(2000);
	cnt128=0;
	SetDisplayCLR(display_alt);
	DisplayAlt();
	while(cnt128!=128)
	{
		zeroWeightSecCnt=0;
		Int8toDisplayHex(127-cnt128,&display_alt[4]);
		doTflags();
		readHX(1);
		wdt_reset();
	}
	DisplayNormal();
	ref=sum128-zero;
	config.cal_denom=ref;
	config.cal_num=refWeights[refIndex]<<7;
	switchAction=SWITCH_NONE;
	ResetTimeout();
	HXdisplayWeight=true;
	while (switchAction!=SWITCH_SHORT)
	{
		zeroWeightSecCnt=0;
		doTflags();
		readHX(1);
		if (secCnt>59)	// user decided to cancel
		{
			PowerOff();
		}
		wdt_reset();
	}
	config.isCalibrated=true;
	eFlags &= ~EFLAG_UNCAL;
	WriteConfig();
}

int main(void)
{
//	Int24toDisplayHex(0x123456,&display[1]);
/*	volatile weight_t w,lastW;
	volatile uint16_t i;
	for (i=0;i<10000;i++)
	{
		w=GetAVG2(1000000l);
		if (w==lastW)
		{
			asm("NOP");	// set a break here and look at i
		}
		lastW=w;
	}*/
	
/*	volatile weight_t w,lastW,t;
	t=100000;
	for (uint8_t i=0;i<100;i++)
	{
		w=GetAVG3(t);
		asm("nop");
	}*/

	Init();
	
	HXdisplayWeight=false;
	DisplayNormal();
/*	SetDisplayRESET(display);
	DelayMs(1000);*/
	SetDisplayAuto0(display);
	DelayMs(1000);
	hxCnt=0;
	SetDisplayCLR(display_alt);
	DisplayAlt();
#if defined(USE_GetAvg1)
	while (hxCnt<=histSize)
#elif defined(USE_GetAvg2)
	while (hxCnt<=avgFactor*12)
#elif defined(USE_GetAvg3)
	while (hxCnt<=NUM_COEFF*2)
#endif
	{
		Int8toDisplayHex(hxCnt,&display_alt[4]);
		readHX(1);
		wdt_reset();
		sleep_cpu();
	}
	DisplayNormal();
	ResetTimeout();
	displayBrightness=3;
	DisplayNormal();
	weightZero=weightAvg;	// Start with 0 weight
//	DisplayAlt();	// testing, showing VBat and Temp
	displayBlinkMask=0;
	HXdisplayWeight=true;
	if (!config.isCalibrated)
	{
		eFlags|=EFLAG_UNCAL;
	}
	switchAction=SWITCH_NONE;
    while(1)
    {
		readHX(1);
		if (eFlags)
		{
			doEFlags();
		}
		if (tFlags)
		{
			doTflags();
		}
		if (switchAction!=SWITCH_NONE)
		{
			doSwitch();
		}
	wdt_reset();
	sleep_cpu();
    }
}