/*! \file display.c

	\brief	Handles the 7-Segment Display
	
	\copyright Copyright (C) 2018 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
*/

#include <avr/io.h>
#include <stdint.h>
#include "display.h"
#include "scale.h"

/*!	\brief	Display RAM for normal Display display[0] is left

	\note This is one byte larger than the display.
		Negative values will be posted to display[1]
		instead of display[0]
		with display[0] set to DISPLAY_MINUS
		so we need one digit more!
*/
volatile uint8_t	display[7];
volatile uint8_t	display_alt[7];	///< Display RAM for alternate Display display_alt[5] is right
/*!	\brief Selects the Display Mode

	If set to DISPLAY_NORMAL, only display[] is displayed.
	If set to DISPLAY_ALT_SHOW_NORMAL, the timer displays
	display[] and inserts display_alt[] for a shorter time.
	display_alt[] normally contains some error message.
	Do not set it directly.
	Use inline functions DisplayNormal() and DisplayAlt()
	to switch between the modes.
*/
volatile DISPLAY_MODE_t	displayMode;
/*!	\brief	The current Display Brightness

	Values from 0 to 3 are allowed.
	0=display off,
	3=full brightness
*/
volatile uint8_t	displayBrightness;

/*!	\brief	Sets Display Blink Mode.

	May be set to any combination of the BLINK_ flags.
*/
volatile uint8_t	displayBlinkMask;

/*!	\brief	Controlls Leading Zero Blanking

	Set to TRUE before you display a Number with
	leading zero blanking.
*/
bool	lzb;

/*!	\brief	The Character to 7-Segment mapping

	Bit 0 = Segment A<br>
	...<br>
	Bit 6 = Segment G<br>
	Bit 7 = DP<br>
	Set display[] to any of this values or use custom characters.
*/
const uint8_t chars[]= 
{
0b00111111,	// 0
0b00000110,	// 1
0b01011011,	// 2
0b01001111,	// 3
0b01100110,	// 4
0b01101101,	// 5
0b01111101,	// 6
0b00000111,	// 7
0b01111111,	// 8
0b01101111,	// 9
0b01110111,	// A
0b01111100,	// b
0b00111001,	// C
0b01011110,	// d
0b01111001,	// E
0b01110001,	// F
0b00111000,	// L (0x10)
0b01010100, // n (0x11)
0b01010000, // r (0x12)
0b01111000,	// t (0x13)
0b01000000, // - (0x14)
0b01100011, //  (0x15)
0b00000000,	// <space> (0x16)
0b00011100, // u (0x17)
0b01011100, // o (0x18)
0b01011000, // c (0x19)
0b01110100, // h (0x1a)
0b00000010,	// ' (0x1b)
0b01110110,	// H (0x1c)
};
	

/*!	\brief	Advances to the next Digit

	This is done every ms. Since we have 6 digits the
	display refresh rate is 166.7Hz.
	This is fairly high but if dimming is used
	it reduces to 41 Hz allowing 3 different
	brightnesses plus off.
*/
void	NextDigit(void)
{
	static uint8_t digit=6;	// Counter for current digit
	uint8_t data;	// Segment data
	volatile static uint8_t	round;	// counter for display brightness
	
	// Switch all segments and digit drivers off to avoid "shadows"
	PORTB &= 0b00111000;	// Segments A,B,C,G,DP low
	PORTB |= 0b11000111;//!
	DDRB  &= 0b00111000;	// Segments A,B,C,G,DP input
	PORTC &= 0b11100000;	// Segment D, Digit 1-4 low
	DDRC  &= 0b11101111;	// Segment D input
	PORTD &= 0b00111100;	// Segment E,F low, Digit 5-6 low
	DDRD  &= 0b00111111;	// Segment E,F input
	// Set the digit driver
	switch (digit)
	{
		case 6: PORTD |= 0b00000010;	// the rightmost digit
				break;
		case 5: PORTD |= 0b00000001;
				break;
		case 4: PORTC |= 0b00001000;
				break;
		case 3: PORTC |= 0b00000100;
				break;
		case 2: PORTC |= 0b00000010;
				break;
		case 1: PORTC |= 0b00000001;	// the leftmost digit
				break;
	}
	// Get the Data for this digit
	if (displayMode==DISPLAY_ALT_SHOW_ALT)
	{
		data=display_alt[digit-1];
		if ((displayBlinkMask&BLINK_ALT) && ((tickCnt>=35) && (tickCnt<65)))
		{
			data=0;	// blinking phase
		}
	} 
	else
	{
		data=display[digit-1];
		if ((displayBlinkMask&BLINK_FAST) && (tickCnt>=50))
		{
			data=0;
		}
		if ((displayBlinkMask&BLINK_SLOW) && (secCnt&1))
		{
			data=0;
		}
	}
	if (round>=displayBrightness)
	{
		data=0;
	}
	PORTB|=(data & 0b11000111);	// Set lit segments high
	DDRB |=(data & 0b11000111);	// Set lit segments as output
	if ((data&0b00001000)!=0)	// segment D
	{
		PORTC|=0b00010000;
		DDRC |=0b00010000;
	}
	PORTD|= (data<<2) & 0b11000000;
	DDRD |= (data<<2) & 0b11000000;
	// decrement digit counter
	digit--;
	if (digit==0)
	{
		digit=6;
		round++;
		if (round>=3)
		{
			round=0;
		}
	}
}

/*!	\brief	Writes 2 Hex-Digits to the Display RAM
	\param b The Byte
	\param p The address inside the Display RAM
	
	The second digit is only displayed if it's address
	is not display[7] or display_alt[6].
	This allows the display to be shiftet right by one
	to make room for a '-' sign.
*/
void	Int8toDisplayHex(uint8_t b, volatile uint8_t *p)
{
	if ((b & 0xf0) || lzb==false) {
		*p=chars[b>>4];
		lzb=false;
	} else {
		*p=chars[DISPLAY_SPACE];
	}
	p++;
	if ((p== &display[3]) || (p== &display_alt[3]) || (p== &display[4]) || (p== &display_alt[4]))
	{
		lzb=false;	// don't blank the gram-digit and following
	}
//	if ((p!= &display[6]) && (p!= &display_alt[6]))
	{
		if ((b & 0x0f) || lzb==false || (p== &display[5]) || (p== &display_alt[5]))	// don't blank the last zero
		{
			*p=chars[b&0x0f];
			lzb=false;
		} else
		{
			*p=chars[DISPLAY_SPACE];
		}
	}
}

/*!	\brief	Writes 4 Digits to the Display RAM
	\param i The Uint16 (packed BCD or packed HEX)
	\param p The address inside the Display RAM
*/
void	Int16toDisplayHex(uint16_t i,volatile uint8_t *p)
{
	Int8toDisplayHex(i>>8,p);
	Int8toDisplayHex((uint8_t)i,p+2);
}

/*!	\brief	Writes 6 Digits to the Display RAM
	\param l The Uint32 (packed BCD or packed HEX)
	\param p The address inside the Display RAM
	\note The MSB of l is ignored.
*/
void	Int24toDisplayHex(uint32_t l, volatile uint8_t *p)
{
	lzb=true;
	Int8toDisplayHex(l>>16,p);
	Int16toDisplayHex(l,p+2);
}

/*!	\brief Clears the Display
	\param adr The Display Memory. May be display or display_alt.
*/
void SetDisplayCLR(volatile uint8_t *adr)
{
	adr[0]=chars[DISPLAY_SPACE];
	adr[1]=chars[DISPLAY_SPACE];
	adr[2]=chars[DISPLAY_SPACE];
	adr[3]=chars[DISPLAY_SPACE];
	adr[4]=chars[DISPLAY_SPACE];
	adr[5]=chars[DISPLAY_SPACE];
}

/*/*!	\brief Sets the display to RESET
	\param adr The Display Memory. May be display or display_alt.
	
	Writes "rESEt" to the selected display RAM

void SetDisplayRESET(volatile uint8_t *adr)
{
	adr[0]=chars[DISPLAY_r];
	adr[1]=chars[0x0e];
	adr[2]=chars[0x05];
	adr[3]=chars[0x0e];
	adr[4]=chars[DISPLAY_t];
	adr[5]=chars[DISPLAY_SPACE];
}*/

/*!	\brief Sets the display to Auto-0
	\param adr The Display Memory. May be display or display_alt.
	
	Writes "Auto-0" to the selected display RAM
*/
void SetDisplayAuto0(volatile uint8_t *adr)
{
	adr[0]=chars[0x0a];
	adr[1]=chars[DISPLAY_u];
	adr[2]=chars[DISPLAY_t];
	adr[3]=chars[DISPLAY_o];
	adr[4]=chars[DISPLAY_MINUS];
	adr[5]=chars[0];
}

/*!	\brief Sets the display to don't touch ("-touch")

	This is used during calibration while averaging the
	zero weight or reference weight which each take about 13s.
	Display mode is set to normal by this function.
*/
void SetDisplayDontTouch(void)
{
	DisplayNormal();
	display[0]=chars[DISPLAY_MINUS];
	display[1]=chars[DISPLAY_t];
	display[2]=chars[DISPLAY_o];
	display[3]=chars[DISPLAY_u];
	display[4]=chars[DISPLAY_c];
	display[5]=chars[DISPLAY_h];
}

/*!	\brief Sets the display to rEF

	This is used during calibration to request
	a reference weight.
	Display mode is set to normal by this function.
*/
void SetDisplayRef(void)
{
	DisplayNormal();
	display[0]=chars[DISPLAY_r];
	display[1]=chars[0x0e];
	display[2]=chars[0x0f];
	display[3]=chars[DISPLAY_SPACE];
	display[4]=chars[DISPLAY_SPACE];
	display[5]=chars[DISPLAY_SPACE];
}

/*!	\brief Sets the alternate display to LoBat

	This is always done in alternate display mode
	so display mode is set to alternate by this function.
*/
void SetDisplayAltLoBat(void)
{
	DisplayAlt();
	display_alt[0]=chars[DISPLAY_L];
	display_alt[1]=chars[DISPLAY_o];
	display_alt[2]=chars[DISPLAY_SPACE];
	display_alt[3]=chars[0x0b];
	display_alt[4]=chars[0x0a];
	display_alt[5]=chars[DISPLAY_t];
}

/*!	\brief Sets the alternate display to unCal

	This is always done before entering the main loop
	if ERR_UNCAL is set in eFlags
*/
void SetDisplayAltUncal(void)
{
	DisplayAlt();
	display_alt[0]=chars[DISPLAY_u];
	display_alt[1]=chars[DISPLAY_n];
	display_alt[2]=chars[0x0c];
	display_alt[3]=chars[0x0a];
	display_alt[4]=chars[DISPLAY_L];
	display_alt[5]=chars[DISPLAY_SPACE];
}

/*!	\brief Sets the display to dEFCon (default config)

	This is used during calibration to request
	a reference weight.
	Display mode is set to normal by this function.
*/
void SetDisplayDefCon(void)
{
	DisplayNormal();
	HXdisplayWeight=false;
	display[0]=chars[0x0d];
	display[1]=chars[0x0e];
	display[2]=chars[0x0f];
	display[3]=chars[0x0c];
	display[4]=chars[DISPLAY_o];
	display[5]=chars[DISPLAY_n];
}

/*!	\brief Sets the display to CAL (Calibration)

	This is used during calibration to indicate
	The beginning of the procedure
	Display mode is set to normal by this function.
*/
void SetDisplayCAL(void)
{
	HXdisplayWeight=false;
	DisplayNormal();
	HXdisplayWeight=false;
	display[0]=chars[0x0c];
	display[1]=chars[0x0a];
	display[2]=chars[DISPLAY_L];
	display[3]=chars[DISPLAY_SPACE];
	display[4]=chars[DISPLAY_SPACE];
	display[5]=chars[DISPLAY_SPACE];
}

/*!	\brief Sets the display to HI (High Resolution) for two seconds

	A double click is used to change between fast (low resolution)
	and precision (high resolution, but slow).
	This routine is used to indicate high resolution mode.
	It includes a delay for two seconds.
	\note Display is set to normal mode.
*/
void SetDisplayHiRes(void)
{
	HXdisplayWeight=false;
	DisplayNormal();
	HXdisplayWeight=false;
	display[0]=chars[DISPLAY_SPACE];
	display[1]=chars[DISPLAY_SPACE];
	display[2]=chars[DISPLAY_SPACE];
	display[3]=chars[DISPLAY_SPACE];
	display[4]=chars[DISPLAY_H];
	display[5]=chars[0x01];
	DelayMs(2000);
	HXdisplayWeight=true;
}

/*!	\brief Sets the display to LO (Low Resolution) for two seconds

	A double click is used to change between fast (low resolution)
	and precision (high resolution, but slow).
	This routine is used to indicate low resolution (fast) mode.
	It includes a delay for two seconds.
	\note Display is set to normal mode.
*/
void SetDisplayLoRes(void)
{
	HXdisplayWeight=false;
	DisplayNormal();
	HXdisplayWeight=false;
	display[0]=chars[DISPLAY_SPACE];
	display[1]=chars[DISPLAY_SPACE];
	display[2]=chars[DISPLAY_SPACE];
	display[3]=chars[DISPLAY_SPACE];
	display[4]=chars[DISPLAY_L];
	display[5]=chars[0x00];
	DelayMs(2000);
	HXdisplayWeight=true;
}
