﻿/*! \file config.c

	\brief	Routines for handling Configuration Data
	\author Robert Loos <robert.loos@loosweb.de>
	
	\copyright Copyright (C) 2014 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p>The routines here define the configuration memory and setting up the
default configuration.</p>
 */ 

#include <avr/eeprom.h>
#include <stdio.h>
#include <math.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include "scale.h"
#include "config.h"
#include "display.h"

//! The Configuration. It is read on startup from the EEPROM.
confStruct_t	config;

/*!	\brief This struct will be stored in the EEPROM and is contained in
	the Production File.
	
	'New' devices are programmed using the production file which also can
	write the default EEPROM content and the fuses.
*/
confStruct_t	EEMEM eepConfig=
{
	.magic=CONF_MAGIC,	// Magic number to detect unprogrammed devices
	.cal_num=100000,
	.cal_denom=210000,
	.isCalibrated=false,
};

/*!	\brief	Loads default config values into conf-struct and stores it in non volatile Memory

	This function is mostly useful during development but I recommend to leave it functional.
	if you change the config struct always also change CONF_MAGIC. It will reprogram
	the EEPROM with default values then. This prevents surprises with different config-structs.
	Especially in devices already in the field it would lead to unpredictable results
	if the configuration memory format would not be compatible with the current version.
	\note Keep the values here in sync with the values in eepConfig if you don't
	like surprises!
*/
static void DefaultConfig(void)
{
	SetDisplayDefCon();
	DelayMs(2000);	// If you use the production file, EEPROM write may be interrupted by a reset if we don't delay here!
	config.magic=CONF_MAGIC;	// Magic number to detect unprogrammed devices
	config.cal_num=100000;
	config.cal_denom=210000;
	config.isCalibrated=false;
	WriteConfig();
	HXdisplayWeight=true; // switch back to weight display (has been disabled by SetDisplayDefCon)
}

/*!	\brief	Reads the <i>Config</i>-Struct from EEPROM into config.

	Only the <i>config</i>-struct in RAM will be used. Changes in <i>config</i> will only
	become permanent if you use a write-command. There is a <i>configDirty</i>-flag
	that is set by every routine that changes <i>config</i> and is used to remind the
	user to execute a write-command.
*/
void ReadConfig(void)
{
	eeprom_busy_wait();
	eeprom_read_block(&config,&eepConfig,sizeof(confStruct_t));	// Copy the default config to the config hold in RAM
	if (config.magic!=CONF_MAGIC)	// no configuration data in EEPROM
	{
		DefaultConfig();
	}
}

/*!	\brief	Writes the <i>Config</i>-Struct to EEPROM
*/
void WriteConfig(void)
{
	HXdisplayWeight=false;
	for (uint8_t i=0;i<6;i++)
	{
		display[i]|=0x80; // show a progress bar with the dots
		DelayMs(300);
	}
	cli();	// Do not disturb!
	wdt_disable();
	eeprom_busy_wait();
	eeprom_write_block(&config,&eepConfig,sizeof(confStruct_t));	// Copy the configuration data from RAM to EEPROM
	eeprom_busy_wait();
	wdt_reset();
	wdt_enable(WDTO_60MS);
	sei();
	HXdisplayWeight=true;
}

