﻿/*!	\file usartcmd.c
	Handles commands from the serial interface

	\brief	Handles commands from the Host Interface
	
	\copyright Copyright (C) 2009-2014 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 

#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <avr/interrupt.h>
#include "adc.h"
#include "delay.h"
#include "eload.h"
#include "uart.h"
#include "config.h"
#include "lcd.h"
#include "ftest.h"
#include "usartcmd.h"
#include "sensors.h"
#include "spi.h"

static const char helpCalibrate[];
static const char helpDebug[];
static const char helpDumpConf[];
static const char helpFTest[];
static const char helpHelp[];
static const char helpReset[];
static const char helpSetBright[];
static const char helpSetContr[];
static const char helpStatus[];
static const char helpUseRts[];
static const char helpWrite[];

#define MAXPARAMS	5	///< Maximum number of Parameters passed to any Command
static char	*cmd;	///< Pointer to the Command Line
static char	*params[MAXPARAMS];
static uint8_t	nParams;

/*!	\brief Prints error message for superfluous parameters
	\param n The number of <b>extra</b> parameters
*/
static void ExtraIgnored(uint8_t n)
{
	hprintf_P(PSTR("%d extra parameters ignored in command %s\n"),n,cmd);
}

/*!	\brief Prints error message for missing arguments
*/
static void MissingArg(void)
{
	hprintf_P(PSTR("Missing value in command %s\n"),cmd);
}

/*!	\brief	Sets the Current to be drawn from the Source

	\return	0 on success, -1 on error.
*/
static int8_t	CmdCurrent(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetCurrent <I>\nI in Ampere\n"));
		MissingArg();
		return -1;
	}
	double	current=atof(params[0]);
	if (current>4.0)
	{
		current=4.0;
	}
	if (current<0.0)
	{
		current=0.0;
	}
	DACSetAmps(current);
	hprintf_P(PSTR("Current is %5.2f\n"),current);
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}


/*!	\brief Prints the Configuration Memory in a human readable form
	\return always 0
*/
static int8_t	CmdDumpConf(void)
{
	DumpConfig();	// Realized in config.c
	if (nParams>0)
		ExtraIgnored(nParams);
	hprintf_P(PSTR("Have a nice day!\n"));
	return 0;	// always succeeds
}


/*!	\brief Prints Values of System Sensors
*/
static int8_t	CmdStat(void)
{
	uint8_t	i,b;
	PGM_P	p;
	
	hprintf_P(PSTR("\nDisplay Temperature: %6.1fC\nHeat Sink Temperature: %6.1fC\n"),
		TempFromADC(),HSTempFromADC());
	hprintf_P(PSTR("P12: %4.2fV Min: %3.2fV Max: %3.2fV Ripple: %3.2fV\n"),P12FromADC(ADCData.named.ADC_P12),
		P12FromADC(ADCDataMin.named.ADC_P12),P12FromADC(ADCDataMax.named.ADC_P12),
		P12FromADC(ADCDataMax.named.ADC_P12-ADCDataMin.named.ADC_P12));
	hprintf_P(PSTR("5V Supply: %3.2fV Min: %3.2fV Max: %3.2fV Ripple: %3.2fV\n"),P5FromADC(ADCData.named.ADC_P5),
		P5FromADC(ADCDataMin.named.ADC_P5),P5FromADC(ADCDataMax.named.ADC_P5),
		P5FromADC(ADCDataMax.named.ADC_P5-ADCDataMin.named.ADC_P5));
	hprintf_P(PSTR("Time: %2d:%02d:%02d\n"),hrCnt,minCnt,secCnt);
	hprintf_P(PSTR("Host commands lost: %d\n"),hostCommandsLost);
	char	str[20];
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("As:%f Time:%s\n"),ADCGetAs(),str);
	if (eflags)
	{
		hprintf_P(PSTR("Error Flags:%02x\n"),eflags);
		for (i=6,b=64;b!=0;b>>=1,i--)
		{
			if (b&eflags)
			{
				memcpy_P(&p,&eflag_descr[i],sizeof(PGM_P));
				hprintf_P(p);
			}
		}
		eflags=0;	// clear flags. may be they will be set again if the error persists.
	}
	if (nParams>0)
		ExtraIgnored(nParams);
	return 0;
}

/*!	\brief	Sets the Brightness of the LCD Backlight

If the brightness is a pure number, it is treated as an absolute value,
a preceding '+' or '-' increases or decreases the brightness for the
specified value.
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetBright(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetBright <x>\nx=0..255\n"));
		MissingArg();
		return -1;
	}
	int16_t	current=LCDGetBrightness();
	int16_t	param=atoi(params[0]);
	if ((params[0][0]=='+') || (params[0][0]=='-'))	// Relative value
	{
		current+=param;
	} 
	else
	{
		current=param;
	}
	if (current>255)
	{
		current=255;
	}
	if (current<0)
	{
		current=0;
	}
	LCDSetBrightness(current);
	config.LCDBright=current;
	if (nParams>1)	// User also wants to set On-Time
	{
		config.backlightOnTime=atoi(params[1]);
		if (config.backlightOnTime==0)
		{
			config.backlightAlwaysOn=1;
		} 
		else
		{
			config.backlightAlwaysOn=0;
		}
	}
	configDirty=true;
	hprintf_P(PSTR("Brightness is %d\n"),LCDGetBrightness());
	if (nParams>2)
	{
		ExtraIgnored(nParams-2);
	}
	return 0;
}

/*!	\brief	Sets the Contrast of the LCD Backlight

If the contrast is a pure number, it is treated as an absolute value,
a preceding '+' or '-' increases or decreases the contrast for the
specified value.
	\return	0 on success, -1 on error.
*/
static int8_t	CmdSetContrast(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: SetContr <x>\nx=0..255\n"));
		MissingArg();
		return -1;
	}
	int16_t	current=LCDGetContrast();
	int16_t	param=atoi(params[0]);
	if ((params[0][0]=='+') || (params[0][0]=='-'))		// Relative value
	{
		current+=param;
	} 
	else
	{
		current=param;
	}
	if (current>255)
	{
		current=255;
	}
	if (current<0)
	{
		current=0;
	}
	LCDSetContrast(current);
	config.LCDBright=current;
	configDirty=true;
	hprintf_P(PSTR("Contrast is %d\n"),LCDGetContrast());
	if (nParams>1)
	{
		ExtraIgnored(nParams-1);
	}
	return 0;
}

/*!	\brief	Writes the Configuration to the EEPROM
	\return	0 on success, -1 on error.
*/
static int8_t	CmdWrite(void)
{
	if (configDirty)
	{
		WriteConfig();
		hprintf_P(PSTR("Configuration written.\n"));
	} else
	{
		hprintf_P(PSTR("Didn't write anything because configuration has not changed.\n"));
	}
	if (nParams>0)
	{
		ExtraIgnored(nParams);
	}
	return 0;
}

void calDisp(void)
{
	while (!(adcFlags&ADCFLAG_NEWAVERAGES))
	{
		wdt_reset();
		sleep_enable();
		sleep_cpu();
		sleep_disable();
	}
	adcFlags&= ~ADCFLAG_NEWAVERAGES;
	LCDGoToXY(0,0);
	printf_P(PSTR("P5:  %4u %6.2fV"),ADCDataAvg.named.ADC_P5,P5FromADC(ADCDataAvg.named.ADC_P5));
	LCDGoToXY(0,1);
	printf_P(PSTR("P12: %4u %6.2fV"),ADCDataAvg.named.ADC_P12,P12FromADC(ADCDataAvg.named.ADC_P12));
	LCDGoToXY(0,2);
	printf_P(PSTR("V:  %5u %6.2fV"),ADCGetVAvgRaw(),ADCGetVavg());
	LCDGoToXY(0,3);
	printf_P(PSTR("I:  %5d %6.2fA"),ADCGetIHighAvgRaw(),ADCGetIavg());
}

/*!	\brief	Receives a Double from the Host
	\return	 The Value
*/
static double	EnterDouble(void)
{
	AssertRTS();
	while (!(cFlags&CFLAG_CMDRECEIVED))
	{
		calDisp();
	}
	cFlags&= ~CFLAG_CMDRECEIVED;
	return atof((char*)cmdline);
}

/*!	\brief	Calibrates the Voltage Sensors

Normal operation is halted during the calibration so
the RTC will be late afterwards.<br>
The user is prompted to enter the voltages from which
the real Volts per LSB for each channel is calculated.
	\note	The JTAGEN-fuse has to be disabled for an accurate Value of P3
and no JTAG debugger must be connected else you would end up in heavy
miscalibration of the P3-value!
	\return	0 on success, -1 on error.
*/
static int8_t	CmdCalibrate(void)
{
	double	d;
	uint8_t	tccr0sav=TCCR0B;
	TCCR0B &= ~0b00000111;	// Stop Timer 0
	LCDClear();
	LCDSetBrightness(0);
	DACSetRaw(0);
	hprintf_P(PSTR("*** Calibration Procedure ***\nHit return to skip a value\n"));
	hprintf_P(PSTR("Please enter the real Value for 5V\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.P5VoltsPerLSB=d/ADCDataAvg.named.ADC_P5;
		configDirty=true;
		hprintf_P(PSTR("P5VoltsPerLSB=%f\n"),config.P5VoltsPerLSB);
	}
	hprintf_P(PSTR("Please enter the real Value for P12\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.P12VoltsPerLSB=d/ADCDataAvg.named.ADC_P12;
		configDirty=true;
		hprintf_P(PSTR("UNregVoltsPerLSB=%f\n"),config.P12VoltsPerLSB);
	}
	hprintf_P(PSTR("Please enter sink voltage\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.ADCDiffVoltsPerLSB=d/ADCGetVAvgRaw();
		configDirty=true;
		hprintf_P(PSTR("ADCDiffVoltsPerLSB=%f %u\n"),config.ADCDiffVoltsPerLSB,ADCGetVAvgRaw());
	}
	SetLowCurrentMode();
	DACSetAmps(0.1);
	do 
	{
		hprintf_P(PSTR("Please reduce the sink voltage to ~5V\nEnter the current in mA\n"));
		calDisp();
		delay1s();
	} while (ADCGetVavg()>5.0);
	d=EnterDouble();
	if (d>0.0)
	{
		config.ADCLowAmpsPerLSB=d/1000.0/ADCGetILowAvgRaw();
		configDirty=true;
		hprintf_P(PSTR("ADCLowMicroAmpsPerLSB=%f %u %u\n"),config.ADCLowAmpsPerLSB*1000000.0,ADCGetILowAvgRaw(),ADCGetIlowLifeRaw());
	}
	DACSetAmps(2.5);
	hprintf_P(PSTR("Enter the current in A\n"));
	d=EnterDouble();
	if (d>0.0)
	{
		config.ADCHighAmpsPerLSB=d/ADCGetIHighAvgRaw();
		configDirty=true;
		hprintf_P(PSTR("ADCLowMilliAmpsPerLSB=%f %u\n"),config.ADCHighAmpsPerLSB*1000.0,ADCGetIHighAvgRaw());
	}
	DACSetAmps(0.0);
	TCCR0B=tccr0sav;	// Restart Timer 0
	InitADC();	// Restore ADC configuration
	LCDClear();
	hprintf_P(PSTR("End calibration procedure.\n"));
	AssertRTS();
	Usart0FlushRx();
	return 0;
}

/*!	\brief	Sets Debug Options

Possible parameters are <i>dcf</i> and <i>sensor</i>. Adding a parameter switches
debug output on, omitting it off. So debug without parameters completely
disables debugging output.
	\note	Be careful using debug output especially for DCF77. This includes calling
interrupt serviced routines (UART) from an interrupt and may lock up if the transmit
buffer is filled (e.g. by auxiliary communication) since in ATMega-architecture
interrupts and non-interruptible!
	\return 0
*/
#ifdef DEBUG
static int8_t	CmdDebug(void)
{
	dFlags=0;
	for (uint8_t i=0;i<nParams;i++)
	{
		if (!strcmp(params[i],"sensor"))
		{
			dFlags|=DFLAG_SENSOR;
		} else
		if (!strcmp(params[i],"adc"))
		{
			dFlags|=DFLAG_ADC;
		} else
		if (!strcmp(params[i],"ah"))
		{
			dFlags|=DFLAG_AH;
		} else
			if (!strcmp(params[i],"rchan"))
		{
		dFlags|=DFLAG_RCHAN;
		}
		else
		{
			hprintf_P(PSTR("Unknown debug flag %s\n\ttry sensor adc ah rchan\n"),params[i]);
		}
	}
	if (dFlags==0)
	{
		hprintf_P(PSTR("Debugging off\nValid flags:\nadc ah rchan sensor\n"));
	}
	return 0;
}
#endif

/*!	\brief	Causes a Watchdog Reset. Useful for Debugging
	\return	This function does not return.
*/
static	int8_t	CmdReset(void)
{
	hprintf_P(PSTR("Bye.\n"));
	for(;;);	// This will lead to a Watchdog Reset
	return 0;	// Never reached
}
static int8_t	CmdHelp(void);

/*!	\brief	Performs a Factory Test
	\return	This function does not return. It ends with a device reset.
*/
static int8_t CmdFTest(void)
{
	
	hprintf_P(PSTR("Factory Test\n"));
	if (FTest())
	{
		hprintf_P(PSTR("*** Test PASSED ***\n"));
	} else
	{
		hprintf_P(PSTR("*** Test FAILED ***\n"));
	}
	return CmdReset();
}

#ifdef USE_RTS
/*!	\brief	Sets if to use RTS/CTS-Handshake.
	\return	0 if OK, -1 on error
*/
static int8_t	CmdUseRts(void)
{
	if (nParams<1)
	{
		hprintf_P(PSTR("Usage: UseRts <x>\nx=[0|1]\n"));
		MissingArg();
		return -1;
	}
	int8_t	useRts=atoi(params[0]);
	config.useRts=(useRts!=0);
	configDirty=true;
	return 0;
}
#endif

static const char helpCalibrate[]	PROGMEM	= "Calibrates the voltage sensors";
#ifdef DEBUG
static const char helpDebug[]		PROGMEM	= "Change debug settings";
#endif
static const char helpDumpConf[]	PROGMEM	= "Prints a human readable list of configuration data";
static const char helpCurrent[]		PROGMEM = "Sets the current to be drawn";
static const char helpFTest[]		PROGMEM	= "Executes the factory test";
static const char helpHelp[]		PROGMEM	= "This list of commands";
static const char helpPreset[]		PROGMEM	= "Sets Preset [1..9] to specified current";
#ifdef DEBUG
static const char helpReset[]		PROGMEM	= "Restarts the device";
#endif
static const char helpSetBright[]	PROGMEM	= "Sets backlight brightness, absolute or relative (e.g. 50, +10 -10)";
static const char helpSetContr[]	PROGMEM	= "Sets LCD contrast, absolute or relative (e.g. 25, +5 -5)";
#ifdef USE_RTS
static const char helpUseRts[]		PROGMEM = "Use RTS/CTS-handshake for serial communication";
#endif
static const char helpStatus[]		PROGMEM	= "Prints current status";
static const char helpWrite[]		PROGMEM	= "Writes configuration to non-volatile memory.\n\
Execute this after any changes in configuration\n or your changes will get lost at next reset!";

static const commandTable_t	commandTable[]= {
	{"Calibrate",CmdCalibrate,helpCalibrate},
	{"Current",CmdCurrent,helpCurrent},
	{"DumpConf",CmdDumpConf,helpDumpConf},
	{"Help",CmdHelp,helpHelp},
	{"FTest",CmdFTest,helpFTest},
	{"SetBright",CmdSetBright,helpSetBright},
	{"SetContr",CmdSetContrast,helpSetContr},
	{"Status",CmdStat,helpStatus},
#ifdef USE_RTS
	{"UseRts",CmdUseRts,helpUseRts},
#endif
	{"Write",CmdWrite,helpWrite},
#ifdef DEBUG
	{"Debug",CmdDebug,helpDebug},
	{"Reset",CmdReset,helpReset},
#endif
	/*	The following are debug-commands that are not officially documented.
		Since they have NULL help texts, they are not listed in the help-command.
	*/
};

/*!	\brief Lists a Description of all Commands
*/
static int8_t	CmdHelp(void)
{
	hprintf_P(PSTR("\nValid commands:\n"));
	for (uint8_t i=0;i<sizeof(commandTable)/sizeof(commandTable_t);i++)
	{
		if (!commandTable[i].helptext)	// no help text, it is a secret command
		{
			continue;
		}
		hprintf_P(PSTR("%10.10s\t"),commandTable[i].cmd);
		hprintf_P(commandTable[i].helptext);
		hprintf_P(PSTR("\n"));
	}
	if (nParams>0)
		ExtraIgnored(nParams);
	return 0;
}


/*!	\brief Executes one command line
	\param str A pointer to the line
	\return 0 on success, 1 on error
*/
uint8_t	ExecCommand(char *str)
{
	cmd=str;
	while (*cmd==' ')
		cmd++;	// skip leading spaces
	if (*cmd=='\0')
	{
		return 1;	// There is no command at all!
	}
	char *cp;
	nParams=0;
	cp=cmd;
	do
	{
		while ((*cp!=' ') && (*cp))
			cp++;	// Advance to next space or end
		if (*cp==' ')
			{
				*cp='\0';	// Terminate the previous param
				cp++;
			}
		while (*cp==' ')	// Skip spaces
			cp++;
		if (*cp)	// Next parameter discovered
		{
			if (*cp=='"')	// It is a String
			{
				cp++;
				char *cp2=strchr(cp,'"');
				if (cp2)	// Hope the string ends somewhere...
				{
					*cp2=0;	// Terminate it
				}
				params[nParams]=cp;
				cp=cp2+1;
			}
			else
			{
				params[nParams]=cp;
			}
			nParams++;
		}
	} while (*cp);
	uint8_t cmdlen=strlen(cmd);
	// Now execute the command
	for (uint8_t i=0;i<sizeof(commandTable)/sizeof(commandTable_t);i++)
	{
		if (strncasecmp(commandTable[i].cmd,cmd,cmdlen)==0)
		{
			commandTable[i].adr();
			if (configDirty)
			{
				hprintf_P(PSTR("* Don't forget to WRITE the configuration to the EEPROM!\n"));
			}
			return 0;
		}
	}
	hprintf_P(PSTR("Error: unknown command \"%s\"\n"),cmd);
	return 1;
}
