/*! \file uart.c

	\brief	Routines for serial Communication
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 


#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <stdio.h>
#include <stdarg.h>
#include <avr/wdt.h>
#include "eload.h"
#include "uart.h"
#include "config.h"

volatile uint8_t	cFlags;
static volatile char	tbuf0[UART0_TXBUFSIZE];	///< Transmit Buffer for UART0
static volatile char	rbuf0[CMDLINE_MAX+1];	///< Receive Buffer for UART0
volatile uint8_t	tbuf0_tail,tbuf0_head;
static volatile uint8_t	rbuf0_ptr;
volatile uint8_t	byteFromHost;	///< Contains the last byte received from the host
volatile uint8_t	hostCommandsLost;	///< Counts the number of commands lost due to buffer overrun

volatile char	cmdline[CMDLINE_MAX+1]; ///< Ready to hold CMDLINE_MAX chars plus terminating 0


FILE uart0=FDEV_SETUP_STREAM(putchar_uart0,NULL,_FDEV_SETUP_WRITE);


//
/*!	\brief	Initialize USART Hardware and Pointers

2 Stop-bits are used,
USART0 connects to the Host at 9600 baud, 8 bits and no parity.
*/
void	InitUart()
{
/*	Initialized to 0 by standard
tbuf0_tail=0;
tbuf0_head=0;
tbuf1_tail=0;
tbuf1_head=0;
rbuf0_ptr=0;
charsInRbuf0=0;
rbuf1_ptr=0;
rbuf0_cksum=0;*/
UBRR0H=(uint8_t)(HOST_UBRR>>8);
UBRR0L=(uint8_t)(HOST_UBRR);
UCSR0B=0b10011000;	// RXCIE TXCIE UDRIE RXEN TXEN UCSZ2 RXB8 TXB8
UCSR0C=0b00000110;	// UMSEL1 UMSEL0 UPM1 UPM0 USBS UCSZ1 UCSZ0 UCPOL
AssertRTS();
}

/*!	\brief	Puts a char to USART0 (Host-Interface)

	\note	This routine blocks when the transmit buffer is full.
	\param	c The Character
	\return	 Always 0
*/
int putc0(char c)
{
uint8_t tmp;

tmp=tbuf0_head+1;
if (tmp==UART0_TXBUFSIZE)
tmp=0;
if ((tmp==tbuf0_tail) && !(SREG&0x80))	// buffer is full and interrupts disabled
{
	return 0;	// discard the character!
}
while (tmp==tbuf0_tail)	// buffer is full. wait...
{
	wdt_reset();
}
cli();
tbuf0[tbuf0_head]=c;
tbuf0_head=tmp;
sei();
UCSR0B|=0b00100000;			// enable UDRE interrupt
return 0;
}

/*!	\brief	The Putchar-Routine for USART0 (Host-Interface)

	\note	This routine blocks when the transmit buffer is full.
	\param	c The Character
	\param	stream A Pointer to the File (unused)
	\return	 Always 0
*/
int putchar_uart0(char c, FILE *stream)
{
	wdt_reset();
	return putc0(c);
}

/*!	\brief	USART0 (Host-Interface) Data Register empty Interrupt
*/
ISR(USART0_UDRE_vect)
{
#ifdef USE_RTS
	if (config.useRts)
	{
		if (UART0_PIN & UART0_CTS_MASK)
		{
			UCSR0B &= ~0b01100000;	// no more interrupts (UDRE and TX to keep the line driver enabled)
			return;	// Interrupts will be re-enabled by timer callback when CTS goes high
		}
	}
#endif
	UDR0=tbuf0[tbuf0_tail];
	tbuf0_tail++;
	if (tbuf0_tail==UART0_TXBUFSIZE)
		tbuf0_tail=0;
	if (tbuf0_head==tbuf0_tail)	// last character in buffer
		UCSR0B &= ~0b00100000;	// no more UDRE interrupts
}

void	CopyRbuf0ToCmdline(void)
{
	uint8_t	tmp;
	
	for (tmp=0;tmp<rbuf0_ptr;tmp++) {
		cmdline[tmp]=rbuf0[tmp];
	}
	cmdline[tmp]=0;	// terminating zero
}

/*!	\brief	USART0 Receive-Interrupt (Host-Interface)

This routine collects incoming characters until a complete
host- or Bluetooth-command has been received.
	\note	For Bluetooth, we ignore the high-byte of the length field. We cannot handle
			Telegrams bigger than the cmdline for now and the do also not appear until now.
*/
ISR(USART0_RX_vect)
{
uint8_t tmp,c;

tmp=rbuf0_ptr+1;
c=UDR0;	// Get character and reset interrupt flag
byteFromHost=c;
cFlags|=CFLAG_BYTERECEIVED;
//UDR0='*';
if ((rbuf0_ptr==0) && (c==10))	// The first character is a linefeed. It is an empty line!
{
	goto excmd;
}
if (c==13)
	return;	// ignore CRs
if (c==10)
{	// LF received. transfer RX-buffer to line buffer and set flag
excmd:
	if (cFlags&CFLAG_CMDRECEIVED)	// Previous command has not been processed yet
	{
		if (hostCommandsLost<255)
		{
			hostCommandsLost++;
		}
	} else	// Copy the received line to the command line
	{
		CopyRbuf0ToCmdline();
		cFlags|=CFLAG_CMDRECEIVED;	// Let main loop know
	}		
	rbuf0_ptr=0;
	rbuf0[0]=0;
	#ifdef USE_RTS
	if (config.useRts)
	{
		ReleaseRTS();	// Keep the host from sending
	}
	#endif
	return;
}
if (tmp==sizeof(rbuf0))	// Buffer wraps on overflow
	tmp=0;
rbuf0[rbuf0_ptr]=c;
rbuf0_ptr=tmp;
}

void Usart0FlushRx(void)
{
	rbuf0_ptr=0;
	cFlags&= CFLAG_BYTERECEIVED|CFLAG_CMDRECEIVED;
}

/*!	\brief	printf_P for Communication with the Host
	\param	fmt Format String
	\param	...	Additional Parameters
	\return	The return value of the underlying vfprintf_P
*/
int hprintf_P(const char *fmt, ... )
{
	int	i;
	va_list	ap;

	va_start(ap,fmt);
	i=vfprintf_P(&uart0,fmt,ap);
	va_end(ap);
	return i;
}