/*!	file spi.h
*/


#ifndef SPI_H_
#define SPI_H_

#include <avr/io.h>
#include <stdbool.h>
#include "uart.h"

#define DACPORT		PORTB
#define DACCSPIN	3
#define ADCPORT		PORTB
#define ADCCSPIN	4

extern uint64_t	ahRaw;
extern uint32_t	ahCnt;
extern uint32_t	ahTickCnt;

extern void	SPIInit(void);
extern void		DACSetRaw(uint16_t value);
extern void		DACSetAmps(double i);
extern void		DACSetAmpsByStr(char *str);
extern void		SpiGetADC(void);
extern void		SpiGetADC2(void);	///< Faster Alternative
extern double	ADCGetIavg(void);
extern double	ADCGetIByVal(uint16_t iRaw, bool isHigh);
extern double	ADCGetAs(void);
extern double	ADCGetVavg(void);
extern uint8_t	SpiSendAndReceiveByte(uint8_t b);
void CalcIMaxLookupTable(void);
extern uint16_t	derating;
extern uint16_t	soaIMaxLUT[128];

typedef struct
{
	uint16_t	V;		///< The Voltage
	uint16_t	Ilow;	///< The Current measured over both Shunts. Invalid if in high Current Range since the 3R3 is shorted by the FET!
	uint16_t	Ihigh;	///< The Current measured over the 100mR Shunt. Always valid but inaccurate if in low current mode.
} SPIADCData_t;

/*!	\brief Raw values for  ADC, DAC and LEDs

	This structure holds the raw values for
	the four channels of an ADC, the actual Value
	for the DAC and LED states.
	It is the direct data stream from the
	ADC and must be filled starting at the end
	to get the correct byte order for an
	uint16_t
*/
typedef struct  
{
	union  {
		struct {
		SPIADCData_t	SPIADCData;
		uint8_t	dummy;	///< Dummy byte
		uint8_t		LED;	///< Bitmask for LED shift register
		uint16_t	DAC; ///< Raw value for the DAC
		uint8_t		DAC_CMD;
		} named;
		char	array[11];
	} data;
}ADC_RcvStruct_t;

/*!	\brief Raw values for  ADC, DAC and LEDs

	This structure holds the raw values for
	the four channels of an ADC, the actual Value
	for the DAC and LED states.
	It is the direct data stream from the
	ADC and must be filled starting at the end
	to get the correct byte order for an
	uint16_t
*/
typedef struct  
{
	union  {
		struct {
			const char ADCCmd[7];
			uint8_t		LED;	///< Bitmask for LED shift register
			uint16_t	DAC; ///< Raw value for the DAC
			uint8_t		DAC_CMD;	///< Command Byte for DAC
		} named;
		char	array[11];
	} data;
}SPI_XmtStruct_t;

extern volatile ADC_RcvStruct_t	ADC_RcvStruct;
extern volatile SPIADCData_t	SPIADCAvg;

/*!	\brief	Returns True if SPI is occupied by ADC

	Since the ADC is handled in the interrupt, no DAC-transfer
	must be made while this function returns true.
	\return	True during ADC-Operations
*/
static inline bool	SPIADCBusy(void)
{
	return (ADCPORT&(1<<ADCCSPIN))==0;
}

/*!	\brief	Asserts DAC CS
*/
static inline void SpiSelectDAC(void)
{
	DACPORT &= ~(1<<DACCSPIN);
}

/*!	\brief	Revoke DAC CS
*/
static inline void SPIDeselectDAC(void)
{
	DACPORT|=(1<<DACCSPIN);
}

/*!	\brief	Asserts ADC CS
*/
static inline void SpiSelectADC(void)
{
	ADCPORT &= ~(1<<ADCCSPIN);
}

/*!	\brief	Revoke ADC CS
*/
static inline void SPIDeselectADC(void)
{
	ADCPORT|=(1<<ADCCSPIN);
}

/*!	\brief	Returns True if High Current Mode is active

	\note	We use PORTD instead of PIND since the gate
	capacitance of the FET prevents PIND from changing
	'instantaneously'.
\return True if High Current Mode is active, False if in Low Current Mode
*/
static inline bool IsHighCurrentMode(void)
{
	bool	isHigh=(PORTD & 0b00000100)!=0;
//	hprintf_P(PSTR("IsHigh: %d\n"),isHigh);
	return (isHigh);
}

extern void SPISetLEDs(uint8_t mask);
extern void	SetLowCurrentMode(void);
extern void	SetHighCurrentMode(void);
extern uint16_t	ADCGetIhighLifeRaw();
extern uint16_t	ADCGetIlowLifeRaw();
extern uint16_t	ADCGetVLifeRaw();
extern uint16_t	ADCGetIHighAvgRaw();
extern uint16_t	ADCGetILowAvgRaw();
extern uint16_t	ADCGetVAvgRaw();
extern void	CalcRChannel(void);

#endif /* SPI_H_ */