/*!	\file spi.c
	\brief	Routines for SPI connected peripherals (ADC, DAC and LEDs)
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 

#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"
#include "Eload.h"
#include "led.h"
#include "spi.h"
#include "screen.h"
#include "uart.h"

/*! \brief Send data buffer for the ADC

	Each conversion needs 3 bytes to send.
	The first one is the command byte followed
	by two "0"-bytes in which the ADC sends its
	result.
	Command byte is composed as:<br>
	S Start bit, always 1<br>
	A2-A0 Channel address<br>
	X Dummy bit<br>
	SGL/DIF 1 for single ended, 0 for differential<br>
	PD1-0 Power down mode<br>
	Simply by sending 7 Bytes all conversions
	for the three channels we use are done.
	The last byte is a dummy byte necessary to
	get the last ADC result.
*/
//uint8_t	SPISendData[7]={0x00,0x00,0xa3,0x00,0x93,0x00,0x83};	///< Commands to the ADC. Is send from high to low index
SPI_XmtStruct_t	SPISendData={{.array={0x00,0x00,0xa3,0x00,0x93,0x00,0x83}}};
volatile ADC_RcvStruct_t ADC_RcvStruct;
volatile SPIADCData_t	SPIADCAvg;	///< Averaged Values over 16 measurements. These values may get up to 14 bits of precision.
volatile uint8_t	SPIPtr;	///< Is set by SPIStartADC() and counted down to zero by the ADC Interrupt
uint16_t	derating;	///< Temperature Derating factor *65535. Maximum Power has to be derated by multiplication with derating.

/*!	\brief	Lookup Table for maximum DAC-Value versus ADC-Voltage

	This table contains the maximum DAC-Values for the high byte of a given ADC voltage result.
	So the ADC interrupt can easily decide to reduce the current with each measurement if the
	SOA is violated.<br>
	This table <b>has to be calculated</b> by CalcIMaxLookupTable() on system startup!
	\note	For now, we keep this as uint16_t but 256 values with two bytes each make a quarter
	of the controllers RAM. We could reduce this to only the upper byte of the DAC at the
	cost of usable SOA area. Further reduction is possible by using less bits of the
	ADC value.<br>
	Note also that the values still have to be degraded according to the case (heatsink) temperature!
*/
uint16_t	soaIMaxLUT[128];

/*!	\brief	Calculates the Lookup Table for IMax
*/
void CalcIMaxLookupTable(void)
{
	double	IMax;
	double	dac;
	
	for (uint16_t i=0;i<128;i++)
	{
		IMax=SOAGetImax((i<<9)*config.ADCDiffVoltsPerLSB);
		dac=IMax/config.DACHighAmpsPerLSB;
		if (dac>65535.0)	// Should never happen!
		{
			dac=65535.0;
		} else if (dac<0.0)	// Should never happen!
		{
			dac=0.0;
		}
		soaIMaxLUT[i]=dac;
//		hprintf_P(PSTR("i:%d V:%f IMax:%f DAC:%f lut:%u\n"),i,(i<<9)*config.ADCDiffVoltsPerLSB,IMax,dac,soaIMaxLUT[i]);
	}
}


/*!	\brief	Sets the State of the 8 LEDs on the SPI-Shift-Register
	\param	mask A Bitmask of the LEDs
	\note	The LEDs are not set immediately but on the next
		SPI interrupt covering the LED shift register.
		This may introduce a small delay.
*/
void SPISetLEDs(uint8_t mask)
{
	SPISendData.data.named.LED=mask;
}

/*!	\brief	Sets Low Current Mode
*/
void	SetLowCurrentMode(void)
{
	//	hprintf_P(PSTR("SLCM\n"));
	PORTD&= ~0b00000100;
	ledData.named.rangeBlue=LED_BLINK_OFF;
	ledData.named.rangeGreen=LED_BLINK_ON;
}

/*!	\brief	Sets High Current Mode
*/
void	SetHighCurrentMode(void)
{
	//	hprintf_P(PSTR("SHCM\n"));
	PORTD|= 0b00000100;
	ledData.named.rangeBlue=LED_BLINK_ON;
	ledData.named.rangeGreen=LED_BLINK_OFF;
}


/*!	\brief	Sends and receives a Byte over the SPI Bus

	SPI implies the reception of a byte whenever
	one is sent. It depends on the hardware if it
	contains useful data.
	\note	This function blocks for the time of the transfer.
\param	b The Byte to transmit
\return	The received Byte
*/
uint8_t	SpiSendAndReceiveByte(uint8_t b)
{
	// We assume SPI is idle!
	SPDR0=b;	// Start Transmission
	while(!(SPSR0 & (1<<SPIF0)));	// Wait for Completion
	return SPDR0;	// Return received Byte and reset Interrupt Flag
}

/*!	\brief	The same as DACSetRaw() but without updating the controls
	This function has been introduced to allow setting the
	DAC from ISRs.
	\note	Fast means the execution time. There is still a delay
	from the call of this function until the DAC is actually set!
\param	value	The Value
*/
inline void	DACSetRawFast(uint16_t value)
{
	SPISendData.data.named.DAC_CMD=0x30;
	SPISendData.data.named.DAC=value;
}

/*!	\brief	Sets the raw Value of the DAC
\param	value	The Value
*/
void	DACSetRaw(uint16_t value)
{
	DACSetRawFast(value);
	hprintf_P(PSTR("DACSetRaw %u\n"),value);
/*	if (IsHighCurrentMode())
	{
		EnterNumberSetValue(&inSink1_s.control->EnterNumber,value*config.DACHighAmpsPerLSB);
	} 
	else
	{
		EnterNumberSetValue(&inSink1_s.control->EnterNumber,value*config.DACLowAmpsPerLSB);
	}*/
}

double	DACAmps;	///< Saves the value last set for the DAC

/*!	\brief	Sets the DAC to a certain Current

	This function also switches the current range to high or low
	if the desired value is higher than rangeHighLimit or lower
	than rangeLowLimit.
	\param	i The Current
*/
void	DACSetAmps(double i)
{
	double	raw;
	
	ClearOvertempError();
	if (i>config.soa.iMax)	// Limit current to maximum allowable
	{
		i=config.soa.iMax;
		SetSOAError();
	} else
	{
		ClearSOAError();
	}
	DACAmps=i;
	if (i>volatileConfig.rangeHighLimit)
	{
		SetHighCurrentMode();
	} 
	else if(i<volatileConfig.rangeLowLimit)
	{
		SetLowCurrentMode();
	}
	if (IsHighCurrentMode())
	{
		raw=i/config.DACHighAmpsPerLSB*volatileConfig.dacHighAdj+0.5;
	} 
	else
	{
		raw=i/config.DACLowAmpsPerLSB+0.5;
	}
	if (raw>65535)
	{
		raw=65535;
	}
	DACSetRaw((uint16_t)raw);
//	hprintf_P(PSTR("DAC %fA,%f\n"),i,raw);
}

/*!	\brief Sets the sink to the value of a string
	\param str A Pointer to the string containing the value in amperes
*/
void	DACSetAmpsByStr(char *str)
{
	double	value;
//	hprintf_P(PSTR("DACSetAmpsByStr %s\n"),str);
	value=atof(str);
	DACSetAmps(value);
}

/*!	\brief	Returns the averaged Value for Load Current

	This function honors the range setting and returns
	the value measured by the active sense resistor.
	\return The averaged Load Current in A
*/
double	ADCGetIavg(void)
{
	if (IsHighCurrentMode())
	{
		return SPIADCAvg.Ihigh*config.ADCHighAmpsPerLSB;
	} 
	else
	{
		return SPIADCAvg.Ilow*config.ADCLowAmpsPerLSB;
	}
}

/*!	\brief Gets the Ampere Seconds consumed
	\returns The As
*/
double	ADCGetAs(void)
{
	double as;
	
	as=ADCGetIByVal(ahRaw/ahCnt,true)*ahTickCnt/100.0;
	return as;
}

/*!	\brief	Returns the Current for a given Value

	\param	iRaw The raw value from the ADC, normalized to 16 bits
	\param	isHigh True if the value represents a high current mode value
	\return The Load Current in A
*/
double	ADCGetIByVal(uint16_t iRaw, bool isHigh)
{
	if (isHigh)
	{
		return iRaw*config.ADCHighAmpsPerLSB;
	} 
	else
	{
		return iRaw*config.ADCLowAmpsPerLSB;
	}
}

/*!	\brief	Returns the averaged Value for the Sink Voltage
	\note The voltage over the sink may be reduced by cable and/or
	additional resistance and may not reflect the exact value
	of the source voltage.
	\return The Voltage over the Sink in V
*/
double	ADCGetVavg(void)
{
		return SPIADCAvg.V*config.ADCDiffVoltsPerLSB;
}

/*!	\brief	Returns the Life Value for the Current

	The current is measured over the lower of the two shunts
	representing the high current range.
	The value is normalized to 16 bit unsigned int.
	\return The current in the range of 0000 to ffff
*/
uint16_t	ADCGetIhighLifeRaw()
{
	return ADC_RcvStruct.data.named.SPIADCData.Ihigh<<1;
}

/*!	\brief	Returns the Life Value for the Current

	The current is measured over the higher of the two shunts
	representing the low current range.
	Note that this value is garbage if the resistor is
	shorted by the MOSFET.
	The value is normalized to 16 bit unsigned int.
	\return The current in the range of 0000 to ffff
*/
uint16_t	ADCGetIlowLifeRaw()
{
	return ADC_RcvStruct.data.named.SPIADCData.Ilow<<1;
}

/*!	\brief	Returns the Life Value for the Sink Voltage

	The value is normalized to 16 bit unsigned int.
	\return The Voltage over the Sink in the range of 0000 to ffff
*/
uint16_t	ADCGetVLifeRaw()
{
	return ADC_RcvStruct.data.named.SPIADCData.V<<1;
}

/*!	\brief	Returns the averaged Value for the Current

	The current is measured over the lower of the two shunts
	representing the high current range.
	The value is normalized to 16 bit unsigned int.
	\return The current in the range of 0000 to ffff
*/
uint16_t	ADCGetIHighAvgRaw()
{
	return SPIADCAvg.Ihigh;
}

/*!	\brief	Returns the averaged Value for the Current

	The current is measured over the higher of the two shunts
	representing the low current range.
	Note that this value is garbage if the resistor is
	shorted by the MOSFET.
	The value is normalized to 16 bit unsigned int.
	\return The current in the range of 0000 to ffff
*/
uint16_t	ADCGetILowAvgRaw()
{
	return SPIADCAvg.Ilow;
}

/*!	\brief	Returns the averaged Value for the Sink Voltage

	The value is normalized to 16 bit unsigned int.
	\return The Voltage over the Sink in the range of 0000 to ffff
*/
uint16_t	ADCGetVAvgRaw()
{
	return SPIADCAvg.V;
}

/*!	\brief	Initializes the SPI Interface
*/
void	SPIInit(void)
{
	SPCR0=(1<<SPE0)|(1<<MSTR0)|(1<<SPR10);	// Enable SPI Master, Clock/64 i.e. 150us/Byte
	SPSR0=0; //1<<SPI2X0;
	SPIDeselectDAC();
	SPIDeselectADC();
	SpiSelectDAC();
	SpiSendAndReceiveByte(0x6f);	// Select internal Reference
	SpiSendAndReceiveByte(0);
	SpiSendAndReceiveByte(0);
	SPIDeselectDAC();
	SPISendData.data.named.DAC_CMD=0x6f;
	// And start the Interrupt
	SpiSelectDAC();
	SPIPtr=sizeof(SPISendData)-1;
	SPCR0|=(1<<SPIE0);	// Enable SPI Interrupt
	SPDR0=SPISendData.data.array[SPIPtr];
	// Interrupt is started now and keeps enabled free running
}

/*!	\brief	Calculates the maximum allowable Current
	This function is designed to be fast enough to be executed
	from the SPI interrupt. It uses the lookup table
	soaIMaxLUT and a fast integer multiplication for
	temperature derating if the current range is high.
	If the actual DAC value is higher than allowed, it is reduced
	and the SOA error is set.
	In low current range no derating is necessary and this
	function does nothing.
*/
static inline void SoaLimit()
{
	if (IsHighCurrentMode())
	{
		uint8_t	i=ADC_RcvStruct.data.named.SPIADCData.V>>8;
		uint16_t lutDerated=((uint32_t)soaIMaxLUT[i]*derating)>>16;
		if (SPISendData.data.named.DAC>lutDerated)
		{
//			DACSetRawFast(lutDerated);
			SPISendData.data.named.DAC=lutDerated;
			SetSOAError();
		}
	}
}

void	CalcRChannel(void)
{
	if (IsHighCurrentMode())	// only in high current mode, when the FET is on!
	{
		uint16_t low=ADCGetILowAvgRaw();
		uint16_t high=ADCGetIHighAvgRaw();
		if ((low>high)&&(high>256))
		{
			double vChannel=(low-high)*SPIADCVOLTSPERLSB;
			double iHigh=high*config.ADCHighAmpsPerLSB;
			volatileConfig.RchanP=vChannel/iHigh;
			volatileConfig.dacHighAdj=(R23+volatileConfig.RchanP)/R23;
			if (dFlags&DFLAG_RCHAN)
			{
				hprintf_P(PSTR("RChanP:%f, vChannel:%f, IHigh:%f, dacHighAdj:%f\n"),volatileConfig.RchanP,vChannel,iHigh,volatileConfig.dacHighAdj);
			}
		}
	}
}

/*!	\brief	The Interrupt Service Routine for SPI

	It collects the Data of the ADC and builds up averages.
	The life 12-bit value read out of the ADC starts at bit 15 in the
	receive struct.
	By averaging 16 values we gain additional 2 bits of accuracy.
	So the average value will be a 16-bit FSR-Value where the 2 LSBs
	are garbage and thus reflect 14 bits.
	It will occur at a frequency of ~12.5kHz with a 7.3728MHz crystal.
	
*/
ISR(SPI_STC_vect)
{
	static SPIADCData_t	avg;
	static uint8_t	avgcnt;
	
	LEDRedOn();
	ADC_RcvStruct.data.array[SPIPtr]=SPDR0;
	if (SPIPtr==offsetof(SPI_XmtStruct_t,data.named.LED))
	{
		SPIDeselectDAC();
		SpiSelectADC();
	}
	if (SPIPtr==0)	// Just got the last Byte
	{
		SPIPtr=sizeof(SPISendData)-1;	// Set Pointer to last Byte
		SPDR0=SPISendData.data.array[SPIPtr];	// And transmit it
		SPIDeselectADC();
		SpiSelectDAC();	// Don't forget, it will be for the DAC
		SoaLimit();
		avg.Ihigh+=ADC_RcvStruct.data.named.SPIADCData.Ihigh>>3;
		avg.Ilow+=ADC_RcvStruct.data.named.SPIADCData.Ilow>>3;
		avg.V+=ADC_RcvStruct.data.named.SPIADCData.V>>3;
		avgcnt++;
		
		if (avgcnt==16)
		{
			avgcnt=0;
			SPIADCAvg.Ihigh=avg.Ihigh;
			SPIADCAvg.Ilow=avg.Ilow;
			SPIADCAvg.V=avg.V;
			ahRaw+=avg.Ihigh;
			ahCnt++;
			avg.Ihigh=0;
			avg.Ilow=0;
			avg.V=0;
		}
	} 
	else
	{
		SPIPtr--;
		SPDR0=SPISendData.data.array[SPIPtr];	// Send next Byte
	}
	LEDRedOff();
}