/*! \file sensors.c

	\brief	Routines for handling Sensor values
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 


#include <inttypes.h>
#include <math.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include "adc.h"
#include "config.h"
#include "led.h"
#include "sensors.h"
#include "spi.h"
#include "uart.h"

/*!	\brief	Converts a P5-Raw ADC Value to Voltage
	\param	val The ADC Value
	\return	The Voltage
*/
double	P5FromADC(uint16_t val)
{
	return val*config.P5VoltsPerLSB;
}

/*!	\brief	Converts a Unreg-Raw ADC Value to Voltage
	\param	val The ADC Value
	\return	The Voltage
*/
double	P12FromADC(uint16_t val)
{
	return val*config.P12VoltsPerLSB;
}

/*!	\brief	Converts the ADC-Temp of the Display NTC Value to Celsius
		This function also calculates the derating factor

	\note	This function takes extensive floating point operations
			and may thus block for about 400us.
	\return	 The Temperature in Celsius
*/
double	TempFromADC(void)
{
	const double rVcc=68000.0;
	const double a1=0.0010721654610564234;
	const double b1=0.0002429457252176066;
	const double c1=0.0;
	const double d1=5.662000988769498e-8;
	const double celsius0=273.15;

	double	untc=ADCData.named.ADC_Temp*(2.56/1024);
	double	p5=P5FromADC(ADCData.named.ADC_P5);
	double	r=rVcc*untc/(p5-untc);
	double	lnr=log(r);
	double	lnr2=lnr*lnr;	// Squared LnR
//	hprintf_P(PSTR("\nTempFromADC: p5:%f, r=%f, untc=%f, lnr=%f\n"),p5,r,untc,lnr);
	double temp= 1/(a1+b1*lnr+c1*lnr2+d1*lnr2*lnr)-celsius0;
	return temp;
}

/*!	\brief	Converts the ADC-Temp Value of the Heat Sink NTC to Celsius

	\note	This function takes extensive floating point operations
			and may thus block for about 400us.
	\return	 The Temperature in Celsius
*/
double	HSTempFromADC(void)
{
	const double rVcc=NTC_RVCC;
	const double a1=NTC_A1;
	const double b1=NTC_B1;
	const double c1=NTC_C1;
	const double d1=NTC_D1;

	double	untc;
	uint16_t	hstemp;
	cli();
	hstemp=ADCData.array[4];
	untc=hstemp*(2.56/1024);
/*	hprintf_P(PSTR("%03d %u %f adcdata: %d %d %d %d %d %d %d %d\n"),tickCnt,hstemp,untc,
		ADCData.array[0],ADCData.array[1],ADCData.array[2],ADCData.array[3],ADCData.array[4],ADCData.array[5],ADCData.array[6],ADCData.array[7]);*/
	sei();
	double	p5=P5FromADC(ADCData.named.ADC_P5);
	double	r=rVcc*untc/(p5-untc);
	double	lnr=log(r);
	double	lnr2=lnr*lnr;	// Squared LnR
//	hprintf_P(PSTR("\nHSTempFromADC: p5:%f, r=%f, untc=%f, lnr=%f\n"),p5,r,untc,lnr);
	double temp= 1/(a1+b1*lnr+c1*lnr2+d1*lnr2*lnr)-celsius0;
	if (temp>config.soa.tempMax)
	{
		DACSetRaw(0);
		SetOvertempError();
		SetLowCurrentMode();
	}
	if (temp<=25.0)
	{
		derating=65535;
	} else
	{
		derating=fabs(1.0-(temp-25.0)/125.0)*65535.0;
	}
/*	uint8_t	i=SPIADCAvg.V>>9;
	uint16_t lutDerated=((uint32_t)soaIMaxLUT[i]*derating)>>16;
	hprintf_P(PSTR("derating: %u (%f)  V:%u  LUT[%u]=%u lutDerated:%u\n"),derating,derating/65535.0,SPIADCAvg.V,i,soaIMaxLUT[i],lutDerated);*/
	return temp;
}

/*!	\brief	Computes the NTC-Resistance for a given Temperature

	\note	This function takes extensive floating point operations
			and may thus block for about 4ms.
	\return	 The Resistance in Ohms
*/
double	RNTCfromTemp(double t)
{
	const double a1=NTC_A1;
	const double b1=NTC_B1;
//	const double c1=NTC_C1;	// Not used in the reverse formula and 0 in our case
	const double d1=NTC_D1;

	t+=celsius0;	// Convert to Kelvin
	double y=(a1-(1/t))/2/d1;
	double x=sqrt( pow( (b1/3/d1),3 ) +y*y );
	double r=exp( pow(x-y,1/3.0) - pow(x+y,1/3.0) );
	return r;
}

/*!	\brief	Computes the ADC-Value for a given Temperature

	\note	This function takes extensive floating point operations
			and may thus block for about 4ms (mostly spent in RNTCfromTemp()).
	\return	 The Temperature in Celsius
*/
uint16_t	ADCfromTemp(double temp)
{
	const double rVcc=NTC_RVCC;
	
	double	r=RNTCfromTemp(temp);
	uint16_t	val=rVcc*(P5FromADC(ADCData.named.ADC_P5)/(r+rVcc)*r) / (config.P5VoltsPerLSB*r);
	return val;
}