﻿/*!	\file screen.c

	\brief	Screen handling
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <avr/pgmspace.h>
#include <avr/sleep.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include "delay.h"
#include "eload.h"
#include "screen.h"
#include "control.h"
#include "lcd.h"
#include "keyboard.h"
#include "menu.h"
#include "sensors.h"
#include "spi.h"
#include "uart.h"
#include "math.h"

#define isEditable(control) ((control)!=CONTROL_TEXTLA && (control)!=CONTROL_REFRESH)

sinkPrefixValues_t	sinkPVal;

volatile uint8_t sFlags;	///< Flags used for Screen Control

/********* Universal buttons ***********/

static const char	TXT_CANCEL[] PROGMEM = "Cancel";	///< The Cancel-Button usually appears in the last line on the left
ctlButton_t	buttonCancel={CONTROL_BUTTON,LCD_LINE4,TXT_CANCEL};
screenItem_t	buttonCancel_s={(control_t*)&buttonCancel,CANCEL};	///< Definition of the standard Cancel Button, last Line, left
	
static const char	TXT_ENTER[] PROGMEM = "Enter";	///< The Enter-Button usually appears in the last line on the right
ctlButton_t	buttonEnter={CONTROL_BUTTON,LCD_LINE4+10,TXT_ENTER};
screenItem_t	buttonEnter_s={(control_t*)&buttonEnter,ENTER};	///< Definition of the standard Enter Button, last line, right

static const char	TXT_BACK[] PROGMEM = "Back";	///< The Back-Button is a Replacement for the Cancel Button
ctlButton_t	buttonBack={CONTROL_BUTTON,LCD_LINE4,TXT_BACK};
screenItem_t	buttonBack_s={(control_t*)&buttonBack,BACK};	///< Definition of the Back Button as a Replacement for the Cancel Button

static const char	TXT_BACKs[] PROGMEM = "<";	///< The small Back-Button is a space-saving alternative
ctlButton_t	buttonBacks={CONTROL_BUTTON,LCD_LINE4+18,TXT_BACKs};
screenItem_t	buttonBacks_s={(control_t*)&buttonBacks,BACK};	///< Definition of the Back Button as a Replacement for the Cancel Button

static const char	TXT_START[] PROGMEM = "Start";	///< The Start-Button usually appears in the 3rd line on the right
ctlButton_t	buttonStart={CONTROL_BUTTON,LCD_LINE3+14,TXT_START};
screenItem_t	buttonStart_s={(control_t*)&buttonStart,START};	///< Definition of the standard Cancel Button, last Line, left

ctlText_t	terminator={CONTROL_TERMINATOR};
screenItem_t	terminator_s={(control_t*)&terminator,0};	///< Definition of the Terminator. This Screen Item <b>has</b> to be the <b>very last</b> Control in a Screen Struct

// Screen Items for Version Screen
/*!	\brief	Prints the Flags set in MCUSR
	\param	param Not used
*/
void	drawResetCause(uint8_t param)
{
	if (mcusrMirror&0x10)
	{
		LCDWriteString_P(PSTR("JTAG "));
	}
	else
	{
		LCDWriteString_P(PSTR("jtag "));
	}
	if (mcusrMirror&0x08)
	{
		LCDWriteString_P(PSTR("WDR "));
	}
	else
	{
		LCDWriteString_P(PSTR("wdr "));
	}
	if (mcusrMirror&0x04)
	{
		LCDWriteString_P(PSTR("BO "));
	}
	else
	{
		LCDWriteString_P(PSTR("bo "));
	}
	if (mcusrMirror&0x02)
	{
		LCDWriteString_P(PSTR("EXT "));
	}
	else
	{
		LCDWriteString_P(PSTR("ext "));
	}
	if (mcusrMirror&0x01)
	{
		LCDWriteString_P(PSTR("PO "));
	}
	else
	{
		LCDWriteString_P(PSTR("po "));
	}
}

static const char	TXT_VERSIONL1[] PROGMEM = "Eload Rev 1.0";	///< The Program Version as to be displayed in the Greeting Screen
static const char	TXT_BUILD[] PROGMEM = "Build " __DATE__;
static const char	TXT_WWW[] PROGMEM ="www.loosweb.de";
ctlText_t	versionL1={CONTROL_TEXTLA,LCD_LINE1,TXT_VERSIONL1};
screenItem_t	versionL1_s={(control_t*)&versionL1,0};
ctlText_t	versionL2={CONTROL_TEXTLA,LCD_LINE2,TXT_BUILD};
screenItem_t	versionL2_s={(control_t*)&versionL2,0};
ctlRefresh_t	resetCause={CONTROL_REFRESH,LCD_LINE3,drawResetCause,0};
screenItem_t	versionL3_s={(control_t*)&resetCause,0};
ctlText_t	versionL4={CONTROL_TEXTLA,LCD_LINE4,TXT_WWW};
screenItem_t	versionL4_s={(control_t*)&versionL4,0};

/*! \brief The Version Screen */
screen_t	screenVersion=
{
	0,15,
	{
		&versionL1_s,&versionL2_s,&versionL3_s,&versionL4_s,
		&terminator_s,
	}
};

// Screen items for manual test
/*!	\brief Draws the measured current for one sink
	\param param The parameter given for the callback function (unused)
*/
void	DrawVIcallback(uint8_t param)
{
	double	volts=ADCGetVavg();
	double	amps=ADCGetIavg();
	formatPrefixValue(&sinkPVal.i,amps);
	formatPrefixValue(&sinkPVal.u,volts);
	LCDWriteString(sinkPVal.u.value);
	LCDWriteString_P(PSTR("V "));
	LCDWriteString(sinkPVal.i.value);
	LCDWriteData('A');
}

/*!	\brief Draws voltage, power and resistance for one sink
	\param param The parameter given for the callback function (unused)
*/
void	DrawPRcallback(uint8_t param)
{
	double	volts=ADCGetVavg();
	double	amps=ADCGetIavg();
	formatPrefixValue(&sinkPVal.p,volts*amps);
	formatPrefixValue(&sinkPVal.r,volts/amps);
	LCDWriteString(sinkPVal.p.value);
	LCDWriteString_P(PSTR("W "));
	LCDWriteString(sinkPVal.r.value);
	LCDWriteData('\3');
}

void	DrawTempCallback(uint8_t param)
{
	printf_P(PSTR("TIC:%4.1f THS:%5.1f"),TempFromADC(),HSTempFromADC());
}

ctlRefresh_t	valueI1={CONTROL_REFRESH,LCD_LINE2,DrawVIcallback,0};
screenItem_t	valueI1_s={(control_t*)&valueI1,0};
ctlRefresh_t	valueUPR1={CONTROL_REFRESH,LCD_LINE3,DrawPRcallback,0};
screenItem_t	valueUPR1_s={(control_t*)&valueUPR1,0};
ctlRefresh_t	temp={CONTROL_REFRESH,LCD_LINE4,DrawTempCallback,0};
screenItem_t	temp_s={(control_t*)&temp,0};

void	EnterSink(CONTROL_RET action,ctlEnterNumber_t *en,uint8_t param);
enterNumberVariable_t	inSink1ev={"0.0000","A"};
ctlEnterNumber_t	inSink1={CONTROL_ENTERNUMBER,LCD_LINE1+11,EnterSink,0,&inSink1ev};
screenItem_t	inSink1_s={(control_t*)&inSink1,0};

/*!	\brief Sets a sink to the Value of an enterNumber-control
	\param action The return value of the enterNumber (not used)
	\param en A pointer to the enterNumber-control
	\param param (unused)
*/
void	EnterSink(CONTROL_RET action,ctlEnterNumber_t *en,uint8_t param)
{
	DACSetAmpsByStr(en->var->value);
}

const char	txtSetPoint_t[] PROGMEM = "Set point:";	///< The Text for Set Point
ctlText_t	txtSetPoint={CONTROL_TEXTLA,LCD_LINE1,txtSetPoint_t};
screenItem_t	txtSetPoint_s={(control_t*)&txtSetPoint,0};

screen_t	screenManualTest=
{
	0,0,
	{
		&txtSetPoint_s,
		&inSink1_s,
		&buttonBacks_s,
		&valueI1_s,&valueUPR1_s,&temp_s,
		&terminator_s
	}
};

uint8_t	doScreenManualTest(void)
{
	return Screen(&screenManualTest);
}
/*************************** End of screen manual test */

void	DrawASCallback(uint8_t param)
{
	double as;
	
	as=ADCGetAs();
	if (as<999.0)
	{
		printf_P(PSTR("%5.1fAs"),as);
	} 
	else if (as<35999)
	{
		printf_P(PSTR("%4.0fmAh"),as/3.6);
	} 
	else
	{
		printf_P(PSTR("%5.1fAh"),as/3600.0);
	}
}


const char	txtCurrent_t[] PROGMEM = "Current:";	///< The Text for Set Point
ctlText_t	txtCurrent={CONTROL_TEXTLA,LCD_LINE1,txtCurrent_t};
screenItem_t	txtCurrent_s={(control_t*)&txtCurrent,0};
ctlRefresh_t	valueAS={CONTROL_REFRESH,LCD_LINE3,DrawASCallback,0};
screenItem_t	valueAS_s={(control_t*)&valueAS,0};

screen_t	screenDischgTest=
{
	0,0,
	{
		&txtCurrent_s,
		&valueAS_s,
		&inSink1_s,
		&buttonBacks_s,
		&buttonStart_s,
		&valueI1_s,&temp_s,
		&terminator_s
	}
};

uint8_t	doScreenDischgTest(void)
{
	uint8_t	action;
	
	while ((action=Screen(&screenDischgTest))!=BACK)
	{
		if (action==START)
		{
			cli();
			ahTickCnt=0;
			ahRaw=0;
			ahCnt=0;
			sei();
		}
	}
	return action;
}
/*************************** End of screen dischg test */

/*!	\brief (Re-) draws a control
	\param control a pointer to the control
	\param mode the drawing mode
*/
void	DrawControl(control_t *control,drawMode_t mode)
{
	LCDGoTo(control->text.pos);	// coordinates are same for all types
	if (isEditable(control->text.type))
	{
		switch (mode)
		{
			case CONTROL_DRAW_MODE_NORMAL:
				LCDWriteData(' ');
				break;
			case CONTROL_DRAW_MODE_EDIT:
				LCDWriteData('#');
				break;
			case CONTROL_DRAW_MODE_SELECT:
				LCDWriteData('*');
				break;
		}	
	}
	switch (control->text.type)
	{
		case CONTROL_TEXTLA:
			printf_P(control->text.txtIndex,control->text.nPar);
			break;
		case CONTROL_BUTTON:
			LCDWriteString_P(control->button.txtIndex);
			break;
		case CONTROL_ENTERNUMBER:
			EnterNumber(&control->EnterNumber,mode);
			break;
		case CONTROL_REFRESH:
			control->refresh.callback(control->refresh.param);
			break;
		case CONTROL_TERMINATOR:
			break;
	}		
}

/*!	\brief Redraws all Auto-Refresh items
*/
static void	RefreshAutoItems(screen_t *screen)
{
	for (int8_t i=screen->numberOfItems-1;i>=0;i--)
	{
		control_t	*pvalue=screen->item[i]->control;
		if (pvalue->text.type==CONTROL_REFRESH)
		{
			DrawControl(pvalue,CONTROL_DRAW_MODE_NORMAL);
		}
	}
//	hprintf_P(PSTR("."));
}

/*	\brief Refreshes the next item on the screen
	\param screen a pointer to the screen
	If the type of the element is not REFRESH,
	nothing happens. It needs as many calls to
	this function as the screen has elements to redraw
	the entire screen.
	The advantage of refreshing the entire screen with
	only one call is that keyboard events can be handled
	in between. This makes editing noticeable smoother...

static void refreshNextItem(screen_t *screen)
{
	static uint8_t	i=0;
	
	control_t	*pvalue=screen->item[i]->control;
	if (pvalue->text.type==REFRESH)
	{
		drawControl(pvalue,MODE_NORMAL);
	}
	i++;
	if (i>=screen->numberOfItems)
		i=0;
}*/

/*!	\brief Handles a screen.
	\param screen A pointer to the screen.
	\return The value of the retval of the control clicked.
	
	A screen may have static, editable and autorefresh-objects.
	If the retval of an editable item is !=0
	(normally used only with buttons)
	the screen is terminated when the user presses enter
	whenever the item is selected.
	If a screen has no editable items at all, the
	screen is terminated when the user presses
	enter, too.
*/
uint8_t	Screen(screen_t *screen)
{
	CONTROL_RET	retval;
	uint8_t	activeItem=255;	// mark invalid
	screenTimeout_t	timeoutCnt;	// Is decremented each refresh period and terminates the screen at 0
	bool	hasActiveItems=false;
	
//	LCDSetColor(0x03);
	LCDClear();
	timeoutCnt=screen->timeout;
	for (screen->numberOfItems=0;screen->item[screen->numberOfItems]->control->text.type!=CONTROL_TERMINATOR;screen->numberOfItems++);
	for (int8_t i=screen->numberOfItems-1;i>=0;i--)	// draw all items and set the first selectable active
	{
		control_t	*pvalue=screen->item[i]->control;
		DrawControl(pvalue,CONTROL_DRAW_MODE_NORMAL);
		if (pvalue->text.type==CONTROL_BUTTON || pvalue->text.type==CONTROL_ENTERNUMBER)
		{
			activeItem=i;
			hasActiveItems=true;
		}
	}
	// This is the Screen Display Loop
	// Be aware that it can stop for an arbitrary time if a ENTERNUMBER-control is selected
	for(;;)
	{
		char	keyPressed;
		if (hasActiveItems)
			DrawControl(screen->item[activeItem]->control,CONTROL_DRAW_MODE_SELECT);
		while ((keyPressed=GetKey())==KEY_NONE)
		{
			sleep_cpu();
			wdt_reset();
			if (sFlags&SFLAG_REFRESH)	// Is set three times per second
			{
//				hprintf_P(PSTR("-"));
				sFlags&= ~SFLAG_REFRESH;
				RefreshAutoItems(screen);
				if (timeoutCnt)
				{
					timeoutCnt--;
					if (timeoutCnt==0)
					{
						return 0;
					}
				}
			}
			DoFlags();
		}
//		hprintf_P(PSTR("KP: %02x (%c)\n"),keyPressed,keyPressed);
		switch (keyPressed)
		{
			case KEY_ROTUP:
				if (hasActiveItems)
				{
					DrawControl(screen->item[activeItem]->control,CONTROL_DRAW_MODE_NORMAL);
					do 
					{
						activeItem--;
						if (activeItem==255)
						{
							activeItem=screen->numberOfItems-1;
						}
					} while (!isEditable(screen->item[activeItem]->control->text.type));
					DrawControl(screen->item[activeItem]->control,CONTROL_DRAW_MODE_SELECT);
				}
				break;
			case KEY_ROTDOWN:
				if (hasActiveItems)
				{
				DrawControl(screen->item[activeItem]->control,CONTROL_DRAW_MODE_NORMAL);
				do 
				{
					activeItem++;
					if (activeItem>=screen->numberOfItems)
					{
						activeItem=0;
					}
				} while (!isEditable(screen->item[activeItem]->control->text.type));
				DrawControl(screen->item[activeItem]->control,CONTROL_DRAW_MODE_SELECT);
				}
				break;
			case KEY_ENTER:
				if (!hasActiveItems)
				{
					return keyPressed;
				}
				if (screen->item[activeItem]->retval)
				{
					return screen->item[activeItem]->retval;
				} 
				else if (screen->item[activeItem]->control->EnterNumber.type==CONTROL_ENTERNUMBER)
				{
					screen->item[activeItem]->control->EnterNumber.var->curPos=0;
					EnterNumberSaveValue(&screen->item[activeItem]->control->EnterNumber);
					do // Loop until the ENTERNUMBER exits
					{
						retval=EnterNumber(&screen->item[activeItem]->control->EnterNumber,CONTROL_DRAW_MODE_EDIT);
						LCDGoTo(cursorPos);
						LCDCursorOn();
						if (((retval==CONTROL_CHANGE)||(retval==CONTROL_ENTER)||(retval==CONTROL_CANCEL)) && (screen->item[activeItem]->control->EnterNumber.changeCallback!=0))
						{
							screen->item[activeItem]->control->EnterNumber.changeCallback(CONTROL_CHANGE,
								&screen->item[activeItem]->control->EnterNumber,
								screen->item[activeItem]->control->EnterNumber.param);
						}
						LCDGoTo(cursorPos);
						do // Wait for a Key-Event
						{
							sleep_cpu();
							wdt_reset();
						} while ((tFlags==0)&&(cFlags==0)&&(!keyStat())&&(sFlags==0));
						if (sFlags&SFLAG_REFRESH)
						{
							sFlags&= ~SFLAG_REFRESH;
							RefreshAutoItems(screen);
						}
						DoFlags();
						wdt_reset();
					} while (retval!=CONTROL_CANCEL && retval!=CONTROL_ENTER);
					LCDCursorOff();
					if (retval==CONTROL_CANCEL)
					{
						EnterNumberRestoreValue(&screen->item[activeItem]->control->EnterNumber);
					}
				}
				break;
			case KEY_CANCEL:
				break;
		}
		if ((keyPressed>='0')&&(keyPressed<='9'))	// A Digit Key was pressed -> set Preset Current
		{
			DACSetAmps(config.preset[keyPressed-'0']);
			EnterNumberSetValue(&inSink1_s.control->EnterNumber,config.preset[keyPressed-'0']);
		}
	}	
}

/*!	\brief Formats a value using SI-Prefix Letters.
	\param val a pointer to the prefixValue_t
	\param newVal the (new) value to be formatted
	The newVal is formatted to fit into the value-string
	of the struct. The function uses prefixes like 'kilo'
	or 'micro' to get easily readable values.
	To avoid flickering when the value changes between
	999 and 1000 (999 and 1.00k) a threshold has been
	implemented to keep the same prefix until the value
	goes too far beyond.
*/
void formatPrefixValue(prefixValue_t *val,double newVal)
{
	prefixType	newPrefix=one;
	const double	prefixThreshold[]={900e-6,900e-3,900.0,900e3,900e6};
	const double	prefixMult[]={1e6,1e3,1,1e-3,1e-6};
	const double	prefixChar[]={'\2','m','*','k','M'};	// \2 is the micro-symbol
	
	if (newVal<0.0)
	{
		newVal=0.0;	// we display only positive values
	}
	if ((newVal>999e6)||(isnan(newVal)))
	{
		strcpy(val->value,"-----");
		return;
	} 
	else if (newVal>999e3)
	{
		newPrefix=mega;
	}
	else if (newVal>999.0)
	{
		newPrefix=kilo;
	}
	else if (newVal>999e-3)
	{
		newPrefix=one;
	}
	else if (newVal>999e-6)
	{
		newPrefix=milli;
	}
	else
	{
		newPrefix=micro;
	}
	if (newPrefix<val->lastPrefix)
	{
		if (newVal<prefixThreshold[newPrefix])
		{
			val->lastPrefix=newPrefix;
		}
		else
		{
			newPrefix=val->lastPrefix;
		}
	}
	else
	{
		val->lastPrefix=newPrefix;
	}
	snprintf(val->value,sizeof(val->value),"%f.5",newVal*prefixMult[newPrefix]);
	if (newPrefix!=one)
	{
		val->value[4]=prefixChar[newPrefix];
		if (val->value[3]=='.')	// ugly period before the unit
		{
			val->value[3]=val->value[2];
			val->value[2]=val->value[1];
			val->value[1]=val->value[0];
			val->value[0]=' ';
		}
	}
}