﻿/*! \file menu.c
	\brief Handles menu display and selection
	
	Menus are handled in pages. Due to display
	limitations, each page can have a maximum of
	7 entries. The main menu is called as a
	page. Submenus are subsequent calls to
	menue().
*/ 

#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <stddef.h>
#include "Eload.h"
#include "stdint.h"
#include "stdlib.h"
#include "menu.h"
#include "lcd.h"
#include "keyboard.h"
#include "screen.h"
#include "spi.h"
#include "uart.h"
#include "usartcmd.h"

uint8_t	redrawRequest;	///< Set to 1 by USB if screen was modified

static const char	TXT_BACK[];
static const char	TXT_BACK[] PROGMEM="Back";
static const char	TXT_MAINMENU[];
static const char	TXT_MAINMENU[] PROGMEM=" --- Main Menu ---";

const char TXT_MAIN1[];
const char TXT_MAIN1[] PROGMEM="Manual Test";
const char TXT_MAIN2[] PROGMEM="Discharge Test";
/*const char TXT_MAIN3[] PROGMEM="yyy";
const char TXT_MAIN4[] PROGMEM="zz";
const char TXT_MAIN5[] PROGMEM="1";
const char TXT_MAIN6[] PROGMEM="2";
const char TXT_MAIN7[] PROGMEM="3";*/

/*! \brief The structure of the main menu
*/
const menu_t menuStructMain PROGMEM= {
	TXT_MAINMENU,
	{
		{ 0,TXT_MAIN1,doScreenManualTest },
		{ 0,TXT_MAIN2,doScreenDischgTest },
/*		{ 0,TXT_MAIN2,0 },
		{ 0,TXT_MAIN3,0 },
		{ 0,TXT_MAIN4,0 },
		{ 0,TXT_MAIN5,0 },
		{ 0,TXT_MAIN6,0 },
		{ 0,TXT_MAIN7,0 },*/
		{ 0,NULL,0 }
	}
};

/*!	\brief	Displays the Main Menu
*/
void	MenuMain(void)
{
	Menu(&menuStructMain);
}

uint8_t	activeItem,numberOfItems;

/*! \brief Draws a menu page (without erasing the display before)
*/
static void	MenuDraw(const menu_t *menu)
{
	static int8_t	firstVisibleLine=0;
	
	if (firstVisibleLine+LCD_LINES-2<activeItem)	// Scroll down so that active item disappears over the first line
	{
		firstVisibleLine=activeItem-LCD_LINES+2;
	} 
	else if (firstVisibleLine>activeItem)	//	Scroll Up over active Item
	{
		firstVisibleLine=activeItem;
	}
	LCDClear();
	LCDGoToXY(0,0);
	LCDWriteString_P((const char*)(pgm_read_word((uint16_t)menu+offsetof(menu_t,head))));
	for (uint8_t i=1;i<LCD_LINES;i++)
	{
		const char *str;
		uint8_t	entryIndex=firstVisibleLine+i-1;
		
		if (entryIndex<numberOfItems)
		{
			str=(const char*)(pgm_read_word((uint16_t)menu+offsetof(menu_t,entry[entryIndex].text)));
		} 
		else
		{
			str=PSTR("");
		}
		LCDGoToXY(0,i);
		if (entryIndex==activeItem)
		{
			LCDWriteData('*');
		} 
		else
		{
			LCDWriteData(' ');
		}
		LCDWriteString_P(str);
	}
}

/*! \brief Executes a menu page
	\param menu A pointer to an MENU_t-struct
	
	The display is erased and the menu is drawn.
	If an entry is selected, the according function
	is called.
	This function returns if the cancel-key is pressed.
	On return, the display is cleared.
*/
void	Menu(const menu_t *menu)
{
	const char	*cp;
	
	LCDClear();
	activeItem=0;
	numberOfItems=0;
	do
	{
		cp=(const char *)pgm_read_word((uint16_t)menu+offsetof(menu_t,entry[numberOfItems].text));
		numberOfItems++;
	} while (cp!=NULL);
	numberOfItems--;
	hprintf_P(PSTR("numberOfItems: %d\n"),numberOfItems);
	for (;;)
	{
		uint8_t keyPressed;
		MenuDraw(menu);
		while((keyPressed=GetKey())==KEY_NONE)
		{
			if (redrawRequest)
			{
				redrawRequest=0;
				LCDClear();
				MenuDraw(menu);
			}
			DoFlags();
			wdt_reset();
			sleep_cpu();
		}
		if (keyPressed==KEY_ROTDOWN)
		{
			activeItem++;
			if (activeItem >= numberOfItems)
			{
				activeItem=0;
			}
		}
		if (keyPressed==KEY_ROTUP)
		{
			keyPressed=KEY_NONE;
			if (activeItem > 0)
			{
				activeItem--;
			} else
			{
				activeItem=numberOfItems-1;
			}
		}
		if (keyPressed==KEY_CANCEL)
		{
			keyPressed=KEY_NONE;
			LCDClear();
			return;
		}
		if (keyPressed==KEY_ENTER)
		{
			uint8_t	(*action)();
			
			action=(uint8_t(*)())pgm_read_word((uint16_t)menu+offsetof(menu_t,entry[activeItem].action));
			if (action)
			{
				(*action)();
				LCDClear();
			} else
			{
				LCDClear();
				return;
			}				
		}
		if ((keyPressed>='0')&&(keyPressed<='9'))	// A Digit Key was pressed -> set Preset Current
		{
			DACSetAmps(config.preset[keyPressed-'0']);
			EnterNumberSetValue(&inSink1_s.control->EnterNumber,config.preset[keyPressed-'0']);
		}
	}
}