﻿/*! \file led.h
	\brief	Provides inline Functions for controlling the LEDs
	\copyright Copyright (C) 2013  Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.

<p>Eight LEDs are possible via a cheap shift register controlled by the
SPI bus. It is really cheap, especially cheaper than choosing the next larger controller.
For additional ~20 cents you could add even eight LEDs more...</p>
<p>They indicate normal operation states as well as error conditions.
The color is defined as follows:</p>
<ul>
<li><span style="background-color:#f44; color:#fff">&nbsp;RED&nbsp;</span> means <span style="background-color:#f44; color:#fff">&nbsp;DANGER&nbsp;</span> and is reserved for error messages or critical states of the cirquit. If you see
a red LED, the circuit is experiencing problems and you should consider immediate
shutdown to avoid damage if it is not already too late.</li>
<li><span style="background-color:#0f0">&nbsp;Green&nbsp;</span> generally means O.K.</li>
<li><span style="background-color:#ff0">&nbsp;Yellow&nbsp;</span> indicates something but also is an O.K. situation.</li>
<li><span style="background-color:#48f; color:#fff">&nbsp;Blue&nbsp;</span> is meant to be the same as yellow but may be used to easily distinguish it from a yellow LED
even if the circuit is a few meters away.
Additional colors are possible of course... nowadays you can get turquoise or
violet LEDs as well as different shades of green, yellow or orange...</li>
</ul>
<p>The LEDs are set within the SPI interrupt sequence and show any
changes without noticeable delay. led_state can be changed any time
according to the situation and will be reflected within the next
SPI interrupt cycle.</p>
 <p>Each LED normally is controlled via 10ms timer callback and generates
 a repetitive blinking sequence. This can be up to 32 bits long, just as you
 typedef the LED_BLINK_t.<br>
 uint8_t seems enough to generate three different blink rates (plus
 'off') but if you want to send debug blink codes uint16_t seems
 the minimum (you need a recognizable pause at the end of each
 code!)<br>
 This behavior can be disabled by commenting out the definition of
 LED_USE_TIMER. You can then switch the LED on or off via the
 inline functions LEDRedOn() and LEDRedOff().
 </p>
*/

#ifndef LED_H_
#define LED_H_

#include "Eload.h"
#include "spi.h"

#define LED_USE_TIMER	///< Enables timer controlled blinking
static const uint8_t LED_TIMER_SLOWDOWN=20;	///< Determines how many Timer Callbacks advance to the next bit

#define LEDREDPORT			PORTC
#define LEDREDMASK	0b00000001

/*!	\brief	Typedef for LED flashing

	Every LED_TIMER_SLOWDOWN timer interrupt (i.e. 20x10ms=200ms), the next bit is selected to
	switch the LED on or off.
*/
typedef uint8_t	LED_FLASH_t;

/*! \defgroup LED_Flash_Values LED Flashing values

	Predefined values for LED flashing
	@{*/
static const LED_FLASH_t LED_BLINK_ON =		0b11111111;	///< always on
static const LED_FLASH_t LED_BLINK_OFF =	0b00000000;	///< always off
static const LED_FLASH_t LED_BLINK_SLOW =	0b00001111;	///< slow blinking, period=1.6s
static const LED_FLASH_t LED_BLINK_NORMAL =	0b00110011;	///< normal speed blinking, period=0.8s
static const LED_FLASH_t LED_BLINK_FAST =	0b01010101;	///< fast blinking, period=0.4s
static const LED_FLASH_t LED_BLINK_BEACON =	0b00000001;	///< short pulse, 200ms on, 1.4s off
/*! @} */
typedef union
{
	struct  
	{
		LED_FLASH_t	powerGreen;
		LED_FLASH_t	powerRed;
		LED_FLASH_t	rangeGreen;
		LED_FLASH_t	rangeBlue;
		LED_FLASH_t	overTemp;
		LED_FLASH_t	SOA;
		LED_FLASH_t	fuse;
		LED_FLASH_t	spare;
	} named;
	LED_FLASH_t	array[8];
} LED_DATA_t;
extern LED_DATA_t	ledData;

/*!	\brief	Switches the red LED on */
static inline void LEDRedOn(void)
{
	LEDREDPORT &= ~LEDREDMASK;
}

/*!	\brief	Switches the red LED off */
static inline void LEDRedOff(void)
{
	LEDREDPORT |= LEDREDMASK;
}

/*!	\brief	Sets the SOA Error Flag and the LED */
static inline void SetSOAError(void)
{
	ledData.named.SOA=LED_BLINK_SLOW;
	eflags|=EFLAG_SOA;
}

/*!	\brief	Clears the SOA Error Flag and the LED */
static inline void ClearSOAError(void)
{
	ledData.named.SOA=LED_BLINK_OFF;
	eflags&= ~EFLAG_SOA;
}

/*!	\brief	Sets the Overtemp Error Flag and the LED */
static inline void SetOvertempError(void)
{
	ledData.named.overTemp=LED_BLINK_FAST;
	eflags|=EFLAG_TEMP;
}

/*!	\brief	Clears the Overtemp Error Flag and the LED */
static inline void ClearOvertempError(void)
{
	ledData.named.overTemp=LED_BLINK_OFF;
	eflags&= ~EFLAG_TEMP;
}

extern LED_FLASH_t	ledRedData;
void	LedTimerCallback(void);


#endif /* LED_H_ */