﻿/*! \file lcd.c

	\brief	Routines for handling the LC-Display
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 

#include <stdio.h>
#include <string.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include "delay.h"
#include "lcd.h"
#include "eload.h"
#include "config.h"
#include "uart.h"

#define LCD_EN		3
#define LCD_RW		2
#define LCD_RS		1
#define LCD_PORT	PORTC
#define LCD_PIN		PINC
#define LCD_DDR		DDRC

uint8_t	backlightOnTime;	///< Time in Seconds the LCD-Backlight is already on (topped at 255 Seconds)
uint8_t	lcdVbarChar[LCD_LINES];	///< Holds the characters for Vertical Bar Graph from Line 3 to 0

// setup stream for LCD
int	putchar_LCD(char c, FILE* file);
FILE lcd=FDEV_SETUP_STREAM(putchar_LCD,NULL,_FDEV_SETUP_WRITE);

const uint8_t LCDStartLine[]={LCD_LINE1,LCD_LINE2,LCD_LINE3,LCD_LINE4};

/*!	\brief	Reads a nibble from LCD
	\return	 Character with data in four MSBs, LSBs are zero
*/
char LCDReadNibble(void)
{
	char	c;

	LCD_PORT |= (1<<LCD_RW);			// switch to read mode
	LCD_DDR=0x0f;
	LCD_PORT = LCD_PORT | (1<<LCD_EN);	// enable=1
	asm("NOP");
	c=LCD_PIN & 0xf0;					// get data
	LCD_PORT &= ~(1<<LCD_EN);			// enable=0
	return c;
}

/*!	\brief	Reads a Byte from the LCD
*/
char	LCDRead(void)
{
	char	c;

	c=LCDReadNibble();
	c|=(LCDReadNibble()>>4);
	return c;
}

char	LCDReadCmd(void)
{
	LCD_PORT &= ~(1<<LCD_RS);	// RS=0
	return LCDRead();
}

/*!	\brief	Waits until the LCD is ready.
*/
void	LCDWait(void)
{
//	return;	// useful for simulation
	uint8_t	i;
	
	i=0;
	while (LCDReadCmd() & 0x80)
		{
		wdt_reset();
			i++;
			if (i==0)
			{
				return;	// timeout (mostly useful for simulator)
			}
		}
}

// Writes the higher four bits to the LCD
/*!	\brief	Writes a nibble to the LCD
	\param	c The Character. Only the upper nibble is used
*/
void	LCDWriteNibble(char c)
{
	LCD_PORT = (LCD_PORT & 0x0f) | (c & 0xf0) | (1<<LCD_EN);
	asm("NOP");
	LCD_PORT &= ~(1<<LCD_EN);
	asm("NOP");
}

/*!	\brief	Writes a Byte to the Display

The state of the RS-pin decides weather it is data or command.
	\param	c The Byte
*/
void	LCDWrite(char c)
{
	LCD_PORT &= ~(1<<LCD_RW);	// set write mode
	LCD_DDR=0xff;				// switch to output
	LCDWriteNibble(c);
	LCDWriteNibble(c<<4);
}

/*!	\brief	Writes a Command-Byte to the Display
	\param	c The Byte
*/
void	LCDWriteCmd(char c)
{
	LCDWait();
	LCD_PORT &= ~(1<<LCD_RS);
	LCDWrite(c);
}

/*!	\brief	Writes a Data-Byte to the Display
	\param	c The Byte
*/
void	LCDWriteData(char c)
{
	LCDWait();
	LCD_PORT |= (1<<LCD_RS);
	LCDWrite(c);
}

/*!	\brief	Displays a String
	\param	s The String
*/
void LCDWriteString(const char *s)
{
while(*s)
	LCDWriteData(*s++);
}

/*!	\brief	Displays a String in Flash Memory
	\param	s The String
*/
void LCDWriteString_P(const char *s)
{
while(pgm_read_byte(s))
	LCDWriteData(pgm_read_byte(s++));
}

/*!	\brief	Clears the LCD and sets Cursor to Home Postion
*/
void	LCDClear(void)
{
	LCDWriteCmd(0x01);	// clear
	LCDWriteCmd(0x02);	// home
}

/*!	\brief	Initializes the Character Generator RAM

	This routine basically defines the antenna-character
	(DCF77-symbol), the micro and the Omega. Since a character 0 cannot be used for
	printf, we use character 1 for it.<br>
	The symbols for bargraph display use character codes
	2 to 6.
	Character code 7 is unused until now.
*/
void LCDInitCGRAM()
{
	const uint8_t cgdata[]={
		// The Antenna
		0b00000100,
		0b00001110,
		0b00010101,
		0b00001110,
		0b00010101,
		0b00000100,
		0b00000100,
		0b00011111,
		// The Micro-Symbol
		0b00000000,
		0b00010001,
		0b00010001,
		0b00010001,
		0b00010011,
		0b00011101,
		0b00010000,
		0b00010000,
		// The Omega-Symbol
		0b00001110,
		0b00010001,
		0b00010001,
		0b00010001,
		0b00001010,
		0b00001010,
		0b00011011,
		0b00000000,
		};
	LCDWriteCmd(0x48);	// Set CGRAM address to 8 (we don't use character 0 for now)
	for (uint8_t i=0;i<sizeof(cgdata);i++)
	{
		LCDWriteData(cgdata[i]);
	}
}

/*!	\brief	Initializes the LC-Display
*/
void	LDCInit(void)
{
	LCD_PORT=0;
	LCD_DDR=0xff;
	delay50ms();
	LCD_PORT=0x30|(1<<LCD_EN);
	delay50us();
	LCD_PORT&=~(1<<LCD_EN);
	delay5ms();
	LCD_PORT|=(1<<LCD_EN);
	delay50us();
	LCD_PORT&=~(1<<LCD_EN);
	delay5ms();
	LCD_PORT|=(1<<LCD_EN);
	delay50us();
	LCD_PORT&=~(1<<LCD_EN);
	delay5ms();
	LCD_PORT=0x20|(1<<LCD_EN);	// Enter 4-bit mode
	delay50us();
	LCD_PORT&=~(1<<LCD_EN);
	delay50us();
	LCDWriteCmd(0x28);	// 4-bit, 2 lines, 5x7
	LCDWriteCmd(0b00001100);	// display on, no cursor, no blinking
	LCDWriteCmd(0x06);	// increment, no scroll
	LCDInitCGRAM();
	LCDClear();
	stdout=&lcd;	// Make LCD the standard output
	delay50ms();
}

void	LCDCursorOn(void)
{
	LCDWriteCmd(0b00001110);
}

void	LCDCursorOff(void)
{
	LCDWriteCmd(0b00001100);
}

/*!	\brief	Sets the Cursor Position

			Lines start at 0 and 64 for the 2x16 and
			0,64,20 and 84 for the 4x20-Display
	\param	pos The Position
*/
void	LCDGoTo(uint8_t pos)
{
	LCDWriteCmd(0x80|pos);
}

/*!	\brief	Sets the Cursor Position
	\param	x The Row Number
	\param	y The Line Number
	\return	The Position as raw Display Value
*/
uint8_t LCDGoToXY(uint8_t x, uint8_t y)
{
	if (x>=LCD_WIDTH)
		x=LCD_WIDTH-1;
	if (y>=LCD_LINES)
		y=LCD_LINES-1;
	uint8_t pos=LCDStartLine[y]+x;
	LCDGoTo(pos);
	return pos;
}

/*!	\brief	The Putchar-Routine for the LCD
	\param	c The Character
	\param	file A pointer to the file (unused)
	\return	 Always 0
*/
int	putchar_LCD(char c, FILE* file)
{
	LCDWriteData(c);
	return 0;
}

/*!	\brief	Printf for one complete Line of the Display

	This is a handy printf-like function for outputting display
	content. It ensures that no more characters are written than
	one line can hold and automatically deletes all characters beyond
	the content so that no fragments of the previous content
	remain visible.
	\param	line The line to print on (0-based)
	\param	fmt	A printf-like format string
	\param	...	Optional further parameters
	\return
*/
void LCDPrintLine_P(uint8_t line, const char *fmt, ...)
{
	char	str[LCD_WIDTH+1];	// line buffer
	va_list	ap;
	
	va_start(ap,fmt);
	vsnprintf_P(str, sizeof(str),fmt,ap);
	va_end(ap);
	if(line>=4)
		line=3;
	LCDGoTo(LCDStartLine[line]);
	str[LCD_WIDTH]=0;	// Ensure a terminated string
	LCDWriteString(str);
	for (uint8_t i=strlen(str);i<LCD_WIDTH;i++)
	{
		LCDWriteData(' ');
	}
}

/*!	\brief	Builds a Bar Graph Character for horizontal Bars

	The horizontal bar graph character can be printed as character code 2
	\param	nBars The number of black Columns (0..5)
	\return	True if OK, false if not
*/
bool LCDBuildHBarChar(uint8_t nBars)
{
	uint8_t	lineChar;
	
	if (nBars>5)
	{
		return false;
	}
	lineChar=0xff;	// All black (5 bars)
	for (uint8_t i=0;i<5-nBars;i++)
	{
		lineChar<<=1;
	}
	LCDWriteCmd(0x40+16);	// Set CGRAM address to 16
	for (uint8_t i=0;i<8;i++)
	{
		LCDWriteData(lineChar);
	}
	return true;
}

/*!	\brief	Builds a Character for vertical Bar Graph

	The vertical bar graph character can be printed as character code 3
	\param	nBars The number of black Lines (0..8)
	\return	True if OK, false if not
*/
bool LCDBuildVBarChar(uint8_t nBars)
{
	if (nBars>8)
	{
		return false;
	}
	LCDWriteCmd(0x40+24);	// Set CGRAM address to 24
	for (int8_t i=0;i<8-nBars;i++)
	{
		LCDWriteData(0);	// Write a blank line
	}
	for (int8_t i=0;i<nBars;i++)
	{
		LCDWriteData(0x1f);	// Write a black line
	}
	LCDGoTo(0);	// Else further writes would go to cgram instead of ddram
	return true;
}

/*!	\brief	Builds the Character Array for vertical Bar Graphs
	\param	nBars The number of black Bars
	\return	True if OK, false if not
*/
bool	LCDBuildVBarChars(uint8_t nBars)
{
	uint8_t	nLines=LCD_LINES;
	
	if (nBars>nLines*8)
	{
		return false;
	}
	for (int8_t i=0;i<nLines;i++)
	{
		int8_t	bars=nBars-i*8;	// The number of bars to appear at index i
		if (bars<=0)
		{
			lcdVbarChar[i]=' ';	// No bar at this position
		} else if (bars>=8)
		{
			lcdVbarChar[i]=255;	// All black here
		} else
		{
			LCDBuildVBarChar(bars);
			lcdVbarChar[i]=3;	// Custom character here
		}
	}
	return true;
}

/*!	\brief	Prints a Line of Bar Graph.
		The rest of the line is cleared.
	\param	line	The 0-based Line Number
	\param	val		The number of black Lines to draw.
		This is not a percent value!
		Maximum is five times the width of the display
		(i.e. 100 for 4x20 and 80 for 2x16).
	\return	-1 on Error, 0 if OK
*/
int8_t	LCDHBarGraph(uint8_t line,uint8_t val)
{
	uint8_t	nBlack;	// Number of characters to draw totally black
	uint8_t	last;	// Number of lines in the last character to appear black
	
	if (line>=LCD_LINES)
	{
		return -1;
	}
	if (val>LCD_WIDTH*5)
	{
		return -1;	// The bar is too long for a line
	}
	nBlack=val/5;
	last=val%5;
	LCDBuildHBarChar(last);
	LCDGoTo(LCDStartLine[line]);
	for (uint8_t i=0;i<nBlack;i++)
	{
		LCDWriteData(255);	// Character 255 is predefined as black
	}
	if (nBlack<LCD_WIDTH)
	{
		LCDWriteData(2);	// Write the fractions of a character
	}
	// From here up to display width...
	for (uint8_t i=0;i<LCD_WIDTH-nBlack-1;i++)
	{
		LCDWriteData(' '); // Clear to EOL
	}
	return 0;
}

/*!	\brief	Has to be called once per Second by the surrounding program

	Timing of functions like LCD-Backlight determine on this timing.
	If you call this function at other intervals, times will change
	accordingly.
*/
void	LCDSecondlyCallback(void)
{
	if (LCDGetBrightness()>0)	// LCD backlight is on
	{
		if (backlightOnTime<255)
		{
			backlightOnTime++;
			if ((backlightOnTime>config.backlightOnTime) && !config.backlightAlwaysOn)	// Backlight is on long enough now!
			{
				LCDSetBrightness(0);	// Switch it off
				backlightOnTime=0;		// And reset the on-time
			}
		}
	}
}

void	LCDActivateBacklight(void)
{
	LCDSetBrightness(config.LCDBright);	// Switch backlight on
	backlightOnTime=0;	// And reset the timer
}