// keyboard.h

#define KEYBOARD_WITH_PRESSED	///< Enables the KEY_PRESSED Event
#define KEYBOARD_WITH_REPEAT	///< Enables the KEY_REPEAT Event
#define KEYBOARD_WITH_LONG		///< Enables KEY_PRESSED_LONG Event
#define KEYBOARD_WITH_RELEASE	///< Enables KEY_RELEASED Event
#define KEYBUFSIZE		16		///< The Size of the Keyboad Buffer. It can keep up to KEYBUFSIZE-1 keys pending

static const uint8_t	KEY_REPEAT_DELAY=60;	///< Key Repetition starts after n Calls to scanKeyboard()/KEY_SCANLINES
static const uint8_t	KEY_REPEAT_RATE=5;		///< Key is Repeated every n Calls to scanKeyboard()/KEY_SCANLINES
static const uint8_t	KEY_ACTION_MASK=0xc0;	///< The upper two Bits of a Key are Flags
static const uint8_t	KEY_PRESSED=0;			///< The Key has just been pressed
static const uint8_t	KEY_REPEAT=0x40;		///< The Key is repeated due to long press time
static const uint8_t	KEY_PRESSED_LONG=0x80;	///< The Key is pressed for at least KEY_REPEAT_DELAY
static const uint8_t	KEY_RELEASED=0xc0;		///< The Key has just been released

typedef uint8_t KEY_CHAR_t;

extern KEY_CHAR_t	GetKey(void);

extern volatile unsigned char keyPressed;
extern volatile uint8_t	keyboardLocked;

#define KEYSCANPORT		PORTB
#define KEYSCANDDR		DDRB
#define KEYRETPORT		PORTA
#define KEYRETPIN		PINA
#define KEYRETDDR		DDRA
#define KEY_SCANLINES		3	///< Number of Scan Lines
#define KEY_RETLINES		4	///< Number of Return Lines
#define ROTPORT			PORTD
#define ROTPIN			PIND
#define ROTBUTTON		(PIND & (1<<PIND5))

/*	Definition of Key-Names.

	Caution: We use the two upper bits of a Key to
	indicate KEY_PRESS, KEY_PRESSED_LONG and KEY_RELEASE,
	so all key codes must be below 0x40!
*/
#define KEY_INVALID	1	///< more than one key pressed
#define KEY_ENTER	2	///< The Enter/Yes Key
#define KEY_CANCEL	3	///< The Cancel/No Key
#define KEY_ROTUP	4	///< Rotary Encoder turned right
#define KEY_ROTDOWN	5	///< Rotary Encoder turned left
#define KEY_NONE	6	///< No Key pressed. Due to the nature of scanKeyboard(), this value has to be >3

/*	Definition of Putkey-Return-Values */
typedef enum {
	PUTKEY_DONE,					///< The return value of Putkey() if the Key has been successfully placed in the Keyboard Queue
	PUTKEY_ERROR_BUFFER_FULL,		///< The return value of PutKey() if no space available. The Character has been discarded.
	PUTKEY_ERROR_KEYBOARD_LOCKED	///< The return value of PutKey() if the Keyboard is locked. The Character has been discarded.
	} PutCharReturn_t;

extern uint8_t	bufReadPtr,bufWritePtr;

/*!	\brief	Returns True if Keystrokes are pending
	\return	False if Keyboard Buffer is empty, True if not.
*/
static inline bool	keyStat(void)
{
	return bufReadPtr!=bufWritePtr;
}

extern uint8_t	keynum;

/*! \brief Initializes the keyboard

	Must be called once on startup
*/
static inline void InitKeyboard()
{
KEYSCANDDR|=0b00000111;	// scan lines are output, other input
KEYSCANPORT|=0b00000110;	// and set only first scanline low
KEYRETPORT|=0b00001111;	// enable pullups at return lines
ROTPIN &= 0b11100111;	// set rotary pins as input...
ROTPORT|=0b00011000;	// ...with pullup
keynum=9;	// no key
}

extern uint8_t	rot,lastRot,rotButton,lastRotButton;
extern const char	rotMatrix[16];
extern PutCharReturn_t	PutKey(KEY_CHAR_t c);

/*!	\brief Scans the rotary encoder

	This routine is declared inline since it is only used once in your project
	inside the timer interrupt.<br>
	It is kept separate from scanKeyboard so you can call it asynchronously,
	let's say every 5ms you call scanRotaryEncoder and every other 5ms you call
	scanKeyboard to minimize the timer interrupt load. But both functions have
	to be called at frequent intervals!
*/
static inline void	scanRotaryEncoder(void)
{
	uint8_t	rotIndex;
	
	rot=(ROTPIN & 0b00011000) >> 3;
	rotIndex=rot | (lastRot<<2);
	char	rotChar=pgm_read_byte(&(rotMatrix[rotIndex]));
	if (rotChar!=0)
	{
		PutKey(rotChar);
	}
	lastRot=rot;
	rotButton=ROTBUTTON;
	if (rotButton!=lastRotButton)
	{
		if (rotButton==0)
		{
			PutKey(KEY_ENTER);
		}
	lastRotButton=rotButton;
	}
}

extern uint8_t	scancnt;
extern uint8_t	keyTimer[KEY_RETLINES*KEY_SCANLINES];
extern const char		decodeMatrix[];
extern uint8_t	repeatRateCnt;

/*! \brief Reads the Return Lines and sets the next Scan Line active.

	This routine is declared inline since it is only used once in your project
	inside the timer interrupt.<br>
	If a key is newly pressed
	its code is placed into the keyboard buffer.
	The scan counter is advanced to the next scan line
	and the next scan line is set active.
	So, even with week pull-ups the return lines will
	stabilize until the next call.
	Must be called from 10ms-timer interrupt. */
static inline void scanKeyboard()
{
unsigned char i;
char	c;

/* The following switch() is hardware dependant and converts
the active return line into a number bewteen 0 and KEY_RETLINES
or 255 if no or more than one key is pressed */
switch(KEYRETPIN&0x0f) {
	case 0x0f:	keynum=255;	// All lines high, no key is pressed
			break;
	case 0x0e:	keynum=0;
			break;
	case 0x0d:	keynum=1;
			break;
	case 0x0b:	keynum=2;
			break;
	case 0x07:	keynum=3;
			break;
	default:	keynum=255;	// more than one key pressed. We treat this as if no key is pressed.
	}
if (keynum<KEY_RETLINES) {	// a key is pressed
	keynum+=scancnt<<2;	// add the value of the current scan line
	c=pgm_read_byte(&(decodeMatrix[keynum]));
	if (keyTimer[keynum]==0) {	// key has just been pressed
		PutKey(c|KEY_PRESSED);
	}
	if (keyTimer[keynum]<KEY_REPEAT_DELAY)
		{
			keyTimer[keynum]++;
		}  else
		if (keyTimer[keynum]<=KEY_REPEAT_DELAY)
		{
			keyTimer[keynum]++;
#ifdef KEYBOARD_WITH_LONG
			PutKey(c|KEY_PRESSED_LONG);
#endif
		} else
		{
			{
#ifdef KEYBOARD_WITH_REPEAT
				repeatRateCnt++;
				if (repeatRateCnt>=KEY_REPEAT_RATE)
				{
					repeatRateCnt=0;
					PutKey(c|KEY_REPEAT);
				}
#endif
			}
		}
	}
else if (keynum==255) {
	for (i=scancnt<<2;i<(scancnt<<2)+KEY_RETLINES;i++)
		{
#ifdef KEYBOARD_WITH_RELEASE
			if (keyTimer[i])
			{
				c=pgm_read_byte(&(decodeMatrix[i]));
				PutKey(c|KEY_RELEASED);
			}
#endif
			keyTimer[i]=0;
		}
	}
scancnt++;
if (scancnt>=KEY_SCANLINES)
	scancnt=0;
/* The following lines are hardware dependant and set all
scan lines high and the active one low. */
KEYSCANPORT|=0b00000111;	// all scanlines high
KEYSCANPORT&=~(0b00000001<<scancnt);	// active scanline low
}

