/*! \file ftest.c
	\brief	Contains Factory Test Routines

	\copyright	Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
*/ 

#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include "delay.h"
#include "eload.h"
#include "ftest.h"
#include "uart.h"
#include "adc.h"
#include "led.h"
#include "lcd.h"
#include "sensors.h"

/*!	\brief Ports to test

	You must specify every port in your system in ascending order (even if it has not
	to be tested) and every	bit that is set will be included in the test.<br>
	Only pins configured as output can be tested!
*/
const static ftestPortsStruct	ports[]=
{
	{
		&PORTA,&PINA,&DDRA,0b11111111	// We can override the voltage dividers!
	},
	{
		&PORTB,&PINB,&DDRB,0b10111111	// Spare out MISO, each of the other lines can be switched to output for Factory Test.
	},
	{
		&PORTC,&PINC,&DDRC,0b11111111
	},
	{
		&PORTD,&PIND,&DDRD,0b11100100	// Only Pin 0 is connected to an external Output. PD3 and PD4 may be grounded by the Encoder
	},
};

static  uint8_t port;	///< The Port currently under test
static  uint8_t bit;	///< The Pin of port currently under test. This is a bit mask.
static  uint8_t testPort;	///< The Port tested against current port
static  uint8_t	tpport;	///< The Read-Value of testPort PORT value
static  uint8_t	tppin;	///< The Read-Value of testPort PIN value
static  uint8_t	tptestMask;	///< The Test Mask of testPort

/*!	\brief	Converts a Byte into a binary String
	\param	byte The byte
	\param	str	A String capable of holding at least the 9 characters representing the binary value
	\return	A pointer to a string
	\note	This function is <b>not reentrant</b>!
		The string <b>must</b> be used before the next call
		since its address is static!
		Multiple arguments in a single printf <b>will not work</b>!
		Split them into several printfs and <b>do not</b> call this function
		from ISRs!
*/
static const char *Byte2bin(uint8_t byte,char str[9])
{
	uint8_t	i=0;
	for (uint8_t mask=128; mask!=0; mask>>=1)
	{
		str[i]=(byte&mask)?'1':'0';
		i++;
	}
	str[8]=0;	// Terminating zero
	return str;
}

/*!	\brief	Prints an Error Message for conflicting Pins
*/
static void	PrintError()
{
	char	str[9];
hprintf_P(PSTR("port %c, pin %d, testport %c, readport %s, "),'A'+port,bit,'A'+testPort,Byte2bin(tpport,str));
hprintf_P(PSTR("readpin %s, "),Byte2bin(tppin,str));
hprintf_P(PSTR("tptestmask %s, "),Byte2bin(tptestMask,str));
hprintf_P(PSTR("eor %s\n"),Byte2bin(tpport ^ tppin,str));
}

/*!	\brief	Prints PASS or FAIL
	\param	pass True if passed
*/
static void	PrintPass(bool pass)
{
	if (pass)
	{
		hprintf_P(PSTR("PASS\n"));
	} 
	else
	{
		hprintf_P(PSTR("FAIL\n"));
	}
}

/*!	\brief	Short Circuit Test #1

All pins included in the testMask[] are set to 1.
A single "0" is shifted through all pins, the pin-registers
are read to verify its presence exactly on this position.
	\return	True on test pass, False on fail
*/
static inline bool	FTestWalkingZero()
{
	static uint8_t	nPorts=sizeof(ports)/sizeof(ftestPortsStruct);
	bool	testPassed=true;
	
	hprintf_P(PSTR("Walking zero\n"));
	for (uint8_t i=0;i<nPorts;i++)	// Set all pins to 1
	{
		*(ports[i].port) |= ports[i].testMask;
	}
	for (port=0;port<nPorts;port++)	// All Ports
	{
		for (bit=1;bit!=0;bit<<=1)	// Each Bit of the current Port
		{
			if (!(ports[port].testMask & bit))	// Bit Testable?
				continue;	// No
			*(ports[port].port)=ports[port].testMask & ~bit;	// Clear one bit
			delay50ms();	// Settling time. The test is fast enough even with this.
			for (testPort=0;testPort<nPorts;testPort++)	// Not check weather other ports remain unchanged
			{
				tpport=*(ports[testPort].port);
				tppin=*(ports[testPort].pin);
				tptestMask=ports[testPort].testMask;
				if ((tpport ^ tppin) & tptestMask)
				{
					PrintError();
					testPassed=false;
					wdt_reset();
				}
			}
			*(ports[port].port)=0xff & ports[port].testMask;	// Set the bit again
		}
	}
	PrintPass(testPassed);
	return testPassed;
}

/*!	\brief	Short Circuit Test #2

All pins included in the testMask[] are set to 0.
A single "1" is shiftet through all pins, the pin-registers
are read to verify its presence exactly on this position.
	\return	True on test pass, False on fail
*/
static inline bool	FTestWalkingOne(void)
{
	static uint8_t	nPorts=sizeof(ports)/sizeof(ftestPortsStruct);
	bool	testPassed=true;

	hprintf_P(PSTR("Walking one\n"));
	for (uint8_t i=0;i<nPorts;i++)	// Set all pins to 0
	{
		*(ports[i].port) &= ~ports[i].testMask;
	}
	for (port=0;port<nPorts;port++)
	{
		for (bit=1;bit!=0;bit<<=1)	// Each Bit of the current Port
		{
			if (!(ports[port].testMask & bit))	// Bit Testable?
			continue;	// No
			*(ports[port].port) |= bit;	// Set one bit
			delay50ms();	// Settling time. The test is fast enough even with this.
			for (testPort=0;testPort<nPorts;testPort++)	// Not check weather other ports remain unchanged
			{
				tpport=*ports[testPort].port;
				tppin=*ports[testPort].pin;
				tptestMask=ports[testPort].testMask;
				if ((*ports[testPort].port ^ *ports[testPort].pin) & ports[testPort].testMask)
				{
					PrintError();
					testPassed=false;
					wdt_reset();
				}
			}
			*ports[port].port &= ~bit;	// Set the bit again
		}
	}
	PrintPass(testPassed);
	return testPassed;
}

/*!	\brief	Test Supply Voltages
	\note The calibration procedure has to be done first
	to get accurate results!
	\return	True on test pass, False on fail
*/
static inline bool	FTestSupply(void)
{
	bool	testPassed=true;
	double v;
	
	v=P12FromADC(ADCDataAvg.named.ADC_P12);
	hprintf_P(PSTR("Testing Supply Voltages\nP12: %5.2fV\n"),v);
	hprintf_P(PSTR("Unreg: %5.2fV (%.1f%%)\n"),v,(v/12.0-1.0)*100.0);
	if ((v>11.0) && (v<=13))
	{
		PrintPass(true);
	} else
	{
		testPassed=false;
		PrintPass(false);
	}
	v=P5FromADC(ADCDataAvg.named.ADC_P5);
	hprintf_P(PSTR("5.0V: %5.2fV (%.1f%%)\n"),v,(v/5.0-1.0)*100.0);
	if ((v>5.0*0.95) && (v<5.0*1.05))
	{
		PrintPass(true);
	} else
	{
		testPassed=false;
		PrintPass(false);
	}
	return testPassed;
}

/*!	\brief	Performs a simple Factory Test

	This routine tries to test everything that can be tested without
	user intervention. It cannot detect all possible error conditions
	but will detect a lot of possible short circuits and other errors
	and can give valuable hints to the test personal.<br>
	This is done by performing a "Walking Zero" and a "Walking One"
	test i.e. all pins are set to 1 (or 0) and a single 0 (or 1)
	walks through all pins. As long only the currently tested pin
	is 0, everything is o.k., if another pin goes to 0, you have
	a short between these two pins. If a pin always stays at a
	fixed level, you have a short to VCC or GND.<br>
	A test mask defines which pins have to be tested. Bits set
	to 0 will not be touched and not be tested. Bits set to
	1 will be programmed as output and tested against all other
	1-pins and the supply.<br>
	A few rules have to be obeyed for the factory test to work:
	<ul>
	<li>All special function pins must be set to normal port I/O.
	Disable all timer outputs, SPIs, USARTS etc. except the
	interface to the host.</li>
	<li>Some devices have a DIDR (digital input disable). Do not forget
	to enable digital inputs.</li>
	<li>The host interface pins must be set to 0 and cannot
	be tested. But they are o.k. if the device is able to
	communicate with the host, anyway.</li>
	<li>Lines with Chip Select functions should be left as output
	with Chip Select set to inactive but set to 0 in the test mask.
	So these pins will not be touched and the chips remain disabled
	throughout the test and do not interfere with the test sequence.</li>
	<li>All lines to other chips that are disabled can be set to 1
	if the connected chip leaves it tri state when disabled, even
	if they are input lines normally.</li>
	</ul>
	Since interrupts are disabled in the progress of this function and
	DDRs are modified, the calling routine should perform a watchdog
	reset to restart the device.<br>
	Do not try to restore previous state and continue normal operation,
	you will most likely forget something and there is nothing bad in
	willingly performing a reset...
	\return True if no error detected, False if test fails
*/
bool	FTest(void)
{
	bool	testPassed=true;
	
	// First a few Tests we need the running System for.
	hprintf_P(PSTR("Press pushbutton key to start the test.\nSystem will reboot afterwards.\n"));
	if (!FTestSupply())
	{
		testPassed=false;
	}
	// And now the "destructive" Test. Reduce system Activity to Factory Test Level.
	// Disable all Interrupts that may change DDR or Port Registers!
	// Disable all Peripherals except the USART to the Host!
	TIMSK0=0;				// Disable timer 0 interrupt
	ADCSRA&= 0b11110111;	// Disable ADC interrupt
	SPCR0=0;	// Disable SPI
	// reset special function pins to normal operation
	TCCR1A&=0b11110000;
	TCCR2A=0;	// Disable Output Compare
	DIDR0=0b00000000;	// Enable digital Functions on all ADC Pins
	for (uint8_t i=0;i<sizeof(ports)/sizeof(ftestPortsStruct);i++)	// Set all DDRs to Test Mask
	{
		*(ports[i].ddr)=ports[i].testMask;	// Set Pins to Output
		*(ports[i].port)=ports[i].testMask;	// Enable Pullups
	}
	if (!FTestWalkingZero())
	{
		testPassed=false;
	}
	if (!FTestWalkingOne())
	{
		testPassed=false;
	}
	return testPassed;
}
