﻿/*
 * control.h
 *
 * Created: 22.08.2011 15:57:22
 *  Author: rolo
 */ 


#ifndef CONTROL_H_
#define CONTROL_H_

#include "stdint.h"
#include "lcd.h"

#define CONTROL_ENTER_NUMBER_MAX_SIZE	7	///< Maximum size of enterNumber-string (including the 0-Terminator)

extern bool	cursorOn;	///< True if the Cursor has to be displayed
extern uint8_t	cursorPos;	///< Actual position of the Cursor (0=Left)

/*! \brief Return values for a control
*/
typedef enum {	CONTROL_NOTHING,	///< Nothing happened. Only used with enterNumber
				CONTROL_CANCEL,		///< User canceled the input. You should set it back to the value you saved before calling the control for the first time
				CONTROL_ENTER,		///< User completed the input. You can take its value for serious.
				CONTROL_CHANGE,		///< The value has changed but neither Enter nor Cancel was pressed. Use it to update a DAC for example.
				CONTROL_EXIT,		///< Nothing happened. This is only used with MODE_SELECT
				CONTROL_ERROR		///< An error occurred. Do not use the input value.
			} CONTROL_RET;

typedef enum { CANCEL=1,ENTER,BACK,START } buttonRet_t;
typedef enum {	CONTROL_TEXTLA,	///< Left-adjusted Textvalue
				CONTROL_BUTTON,	///< A Button
				CONTROL_ENTERNUMBER,	///< An enterNumber-type
				CONTROL_REFRESH,	///<	An Item that will be refreshed periodically (for displaying real-time-values)
				CONTROL_TERMINATOR	///<	Indicating the end of the List
			 } controlType_t;

typedef struct enterNumberVariable_s
{
	char	value[CONTROL_ENTER_NUMBER_MAX_SIZE];	///< The five digits plus string terminator
	char	*suffix_s;	///< This String is displayed after the value. Normally the physical unit (e.g. mA)
	char	oldValue[CONTROL_ENTER_NUMBER_MAX_SIZE];	///< This value will be restored if the user hits cancel
	uint8_t	curPos;	///< cursor position (0=left)
}enterNumberVariable_t;

/*! \brief parameter set for enterNumber()
 */
typedef struct enterNumber_s
{
	const controlType_t	type;	///< The type of control
	uint8_t	pos;	///< Position (see also to LCD_LINEn)
	void	(*changeCallback)(CONTROL_RET action,struct enterNumber_s *control,uint8_t param);	///< if not 0, this function is called every time the value changes
	uint8_t	param;	///< Is passed to the callback function
	enterNumberVariable_t	*var;
} ctlEnterNumber_t;

/*!	\brief Variables for a button-control
*/
typedef struct button_s
{
	const controlType_t	type;
	uint8_t	pos;	///< Position (see also to LCD_LINEn)
	const char	*txtIndex;
} ctlButton_t;

/*!	\brief Variables for a text-control
*/
typedef struct text_s
{
	const controlType_t	type;
	uint8_t	pos;	///< Position (see also to LCD_LINEn)
	const char	*txtIndex;
	uint8_t	nPar;	///< Numeric parameter which can be used in the text (%%d)
}ctlText_t;

/*!	\brief Variables for an auto-refresh-control

	The structure provides a callback function
	which has to draw the value. It will be called
	in regular intervals to actualize its content.
*/
typedef struct refresh_s
{
	const controlType_t	type;
	uint8_t	pos;	///< Position (see also to LCD_LINEn)
	void	(*callback)(uint8_t param);	///< Callback function which has to draw the value.
	uint8_t	param;	///< Parameter for the callback function
}ctlRefresh_t;

/*!	\brief The control_t is a union of all possible control types
*/
typedef union
{
	ctlText_t	text;					///< A static Text
	ctlButton_t	button;				///< A Button
	ctlEnterNumber_t	EnterNumber;	///< An input Field for numerical Values
	ctlRefresh_t	refresh;			///< An Auto-Refresh-Item
} control_t;


/*! \brief mode for drawing controls
*/
typedef enum
	{	CONTROL_DRAW_MODE_NORMAL,	///< The control is drawn as static (not selected and not active)
		CONTROL_DRAW_MODE_EDIT,		///< The control is in edit mode
		CONTROL_DRAW_MODE_SELECT		///< The control is drawn as selected. Pressing enter will change to MODE_EDIT
	} drawMode_t;

extern CONTROL_RET	EnterNumber(ctlEnterNumber_t *val,drawMode_t mode);

void	EnterNumberSetValue(ctlEnterNumber_t *ctl,double val);
void	EnterNumberSaveValue(ctlEnterNumber_t *val);
void	EnterNumberRestoreValue(ctlEnterNumber_t *val);

#endif /* IN5DIGIT_H_ */