﻿/*! \file control.c
	\brief Provides an input field for numerical Values

	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
*/ 

#include <stdlib.h>
#include <string.h>
#include <avr/pgmspace.h>
#include "control.h"
#include "keyboard.h"
#include "ctype.h"
#include "lcd.h"
#include "uart.h"

bool	cursorOn;	// Is set when a control wants the cursor to be displayed
uint8_t	cursorPos;	// Is set by the control
static	char	saveval[CONTROL_ENTER_NUMBER_MAX_SIZE];	// The saved actual value to be restored if The user cancels

/*! \brief Draws the Input Field
	\param val Pointer to enterNumber-struct
	\param mode Drawing mode.
*/
static void	EnterNumberDraw(ctlEnterNumber_t *val,drawMode_t mode)
{
	LCDGoTo(val->pos+1);	// x+1 because of the select-character drawn before!
	LCDWriteString(val->var->value);
	LCDWriteString(val->var->suffix_s);
	if (mode==CONTROL_DRAW_MODE_EDIT)
	{
		cursorOn=true;
		cursorPos=val->pos+1+val->var->curPos;
	}
}

/*! \brief Saves the Value of the Input Field
	Purpose is to reset it to the original value when the user
	exits the control with cancel.
	
	This function must be called when the user enters
	the control.

	\param val Pointer to enterNumber-struct
*/
void	EnterNumberSaveValue(ctlEnterNumber_t *val)
{
	strncpy(val->var->oldValue,val->var->value,CONTROL_ENTER_NUMBER_MAX_SIZE);
}

/*! \brief Restores the old Value of the Input Field
	Purpose is to reset it to the original value when the user
	exits the control with cancel.
	
	This function must be called when the user exits
	the control with the cancel key.

	\param val Pointer to enterNumber-struct
*/
void	EnterNumberRestoreValue(ctlEnterNumber_t *val)
{
	strncpy(val->var->oldValue,val->var->value,CONTROL_ENTER_NUMBER_MAX_SIZE);
	EnterNumberDraw(val,CONTROL_DRAW_MODE_NORMAL);
}

/*! \brief Enter a Number via Keyboard and/or Encoder
	\param val Pointer to input data structure
	\param mode The draw mode
	\return see CONTROL_RET

	The value-member must be set to a default value
	before calling this function.
	The value is a pattern consisting of digits and optionally a
	period. The length of the pattern is constant and the period cannot
	be moved.
	The period <b>must not be</b> the very first or last character!
	The user then may modify the value.
	If return is != CONTROL_ENTER the value
	should not be used since it may have been
	erroneously modified by the user.<br>
	If return is CONTROL_CHANGE you may process
	the change but should call the function again
	(with the same val) to allow the user to
	continue with his input.<br>
	To keep this function non-blocking it will return
	CONRTOL_NOTHING if no key- or encoder event is
	pending. Call it again as soon as you find time to
	until it returns CONTROL_ENTER or CONTROL_CANCEL.
	You can specify a change-callback function which is called
	every time the value is changed by the encoder.
*/
CONTROL_RET	EnterNumber(ctlEnterNumber_t *val,drawMode_t mode)
{
	
	uint8_t	len=strlen(val->var->value);	// Get the length of the given value
	for (uint8_t i=0;i<len;i++)	// make sure we have only digits
	{
		if (!isdigit(val->var->value[i])&& val->var->value[i]!='.')
			val->var->value[i]='0';	// default to 0
	}
	val->var->value[CONTROL_ENTER_NUMBER_MAX_SIZE-1]=0;	// Ensure a terminated string
	val->var->oldValue[CONTROL_ENTER_NUMBER_MAX_SIZE-1]=0;
	if (val->var->curPos>CONTROL_ENTER_NUMBER_MAX_SIZE-2)	// Limit cursor position
		val->var->curPos=CONTROL_ENTER_NUMBER_MAX_SIZE-2;
	EnterNumberDraw(val,mode);
	if (mode==CONTROL_DRAW_MODE_NORMAL||mode==CONTROL_DRAW_MODE_SELECT)
	{
		return CONTROL_EXIT;
	}
	char keyPressed=GetKey();
//	hprintf_P(PSTR("len:%d val: %s oldval:%s\n"),len,val->var->value,val->var->oldValue);
	if (keyPressed==KEY_CANCEL)
	{
		if (val->var->curPos>0)
		{
			val->var->curPos--;
			if (val->var->value[val->var->curPos]=='.')	// Skip the period
			{
				val->var->curPos--;
			}
			EnterNumberDraw(val,CONTROL_DRAW_MODE_EDIT);
		} else
		{
			hprintf_P(PSTR("ex1 len:%d val: %s oldval:%s\n"),len,val->var->value,val->var->oldValue);
			strncpy(val->var->value,val->var->oldValue,CONTROL_ENTER_NUMBER_MAX_SIZE);
			EnterNumberDraw(val,CONTROL_DRAW_MODE_NORMAL);
			hprintf_P(PSTR("ex2 len:%d val: %s oldval:%s\n"),len,val->var->value,val->var->oldValue);
			return (CONTROL_CANCEL);
		}
	}
	else if (keyPressed==KEY_ENTER)
	{
		val->var->curPos++;
		if (val->var->value[val->var->curPos]=='.')	// Skip the period
		{
			val->var->curPos++;
		}
		if (val->var->value[val->var->curPos]==0)	// We reached end of template!
		{
			strncpy(val->var->oldValue,val->var->value,CONTROL_ENTER_NUMBER_MAX_SIZE);
			return CONTROL_ENTER;
		}
		EnterNumberDraw(val,CONTROL_DRAW_MODE_EDIT);
	} 
	else if (keyPressed==KEY_ROTUP)
	{
		strncpy(saveval,val->var->value,CONTROL_ENTER_NUMBER_MAX_SIZE);
		uint8_t pos=val->var->curPos;
		val->var->value[pos]++;
		while (val->var->value[pos]>'9')
		{
			val->var->value[pos]='0';
			if (pos==0)
			{
				break;
			}
			pos--;
			if (val->var->value[pos]=='.')	// Skip the period
			{
				pos--;
			}
			val->var->value[pos]++;
		}
		if ((val->var->value[0]=='0') && (saveval[0]=='9'))	// We got a rollover
		{
			strncpy(val->var->value,saveval,CONTROL_ENTER_NUMBER_MAX_SIZE);	// Restore the old value
		}
		return (CONTROL_CHANGE);
	}
	else if (keyPressed==KEY_ROTDOWN)
	{
		strncpy(saveval,val->var->value,CONTROL_ENTER_NUMBER_MAX_SIZE);
		uint8_t pos=val->var->curPos;
		val->var->value[pos]--;
		while (val->var->value[pos]<'0')
		{
			val->var->value[pos]='9';
			if (pos==0)
			{
				break;
			}
			pos--;
			if (val->var->value[pos]=='.')	// Skip the period
			{	
				pos--;
			}
			val->var->value[pos]--;
		}
		if ((val->var->value[0]=='9') && (saveval[0]=='0'))	// We got a rollover
		{
			strncpy(val->var->value,saveval,CONTROL_ENTER_NUMBER_MAX_SIZE);	// Restore the old value
		}
		return (CONTROL_CHANGE);
	}
	else if (keyPressed>='0' && keyPressed<='9')	// A digit key is pressed
	{
		val->var->value[val->var->curPos]=keyPressed;
		if (val->var->curPos<CONTROL_ENTER_NUMBER_MAX_SIZE-2)
		{
			val->var->curPos++;
			if (val->var->value[val->var->curPos]=='.')	// Skip the period
			{
				val->var->curPos++;
			}
		}
		return (CONTROL_NOTHING);	// Digits shall not immediately change value
	}
	return CONTROL_NOTHING;
}

/*!	\brief	Sets the Value
	\param	ctl The address of the EnterNumber struct
	\param	val	The desired Value
*/
void	EnterNumberSetValue(ctlEnterNumber_t *ctl,double val)
{
	char	str[10];
	
	snprintf(str,sizeof(str),"%6.4f",val);
	strcpy(ctl->var->value,str);
//	hprintf_P(PSTR("enterNumberSetValue: %s\n"),str);
}