﻿/*! \file config.h

	\brief	Structures and Constants for Configuration Handling
	
	\copyright Copyright (C) 2013  Robert Loos	<http://www.loosweb.de>
	
	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.<br>
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.<br>
	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef CONFIG_H_
#define CONFIG_H_

#include <stdbool.h>
#include <stdio.h>
#include "eload.h"

#define CONF_MAGIC	0x0151	///< Increment this value if confStruct_t becomes incompatible to previous versions!

// Definition of the Resistors according to the Schematics
#define	R12	8200.0		///< High side Resistor to the 2.5V Reference Voltage
#define R13	2200.0		///< Low side Resistor to the 2.5V Reference Voltage
#define R16	120000.0	///< High Side Voltage Divider Resistor
#define R18	1000.0		///< Low Side Voltage Divider Resistor
#define R23	0.1			///< High Range Shunt Resistor
#define	R25	3.3			///< Low Range Shunt Resistor

#define SPIVREF (2.5/(R12+R13)*R13)	///< Reference voltage for the SPI ADC
#define SPIADCVOLTSPERLSB	(SPIVREF/65536)	///< Native ADC Voltage per LSB
#define SPIDIFFVOLTSPERLSB	(SPIADCVOLTSPERLSB/R18*(R16+R18))	///< Volts per LSB for the differential Voltage Sensor
#define SPIHIGHAMPSPERLSB	(SPIADCVOLTSPERLSB/R23)	///< Amperes per LSB in High Current Range
#define SPILOWAMPSPERLSB	(SPIADCVOLTSPERLSB/R25)	///< Amperes per LSB in Low Current Range

// Presets in A
#define PRESET0	0.0
#define PRESET1	0.001
#define PRESET2	0.0015
#define PRESET3	0.005
#define PRESET4	0.01
#define PRESET5	0.05
#define PRESET6	0.1
#define PRESET7	0.5
#define PRESET8	1.0
#define PRESET9 4.0

/*#define SOA_PMAX	80.0	///< The maximum allowable Power Dissipation
#define SOA_V2	25.0	///< The Voltage at the upper Margin of const Pv
#define SOA_IMAX	4.0		///< The absolute maximum Current
#define SOA_I2	3.2		///< The Current at the upper Margin of const Pv
#define SOA_I3	0.35	///< The allowable Current at 60V*/
#define SOA_PMAX	80.0	///< The maximum allowable Power Dissipation
#define SOA_V2		25.0	///< The Voltage at the upper Margin of const Pv
#define SOA_IMAX	4.0		///< The absolute maximum Current
#define SOA_I2		3.2		///< The Current at the upper Margin of const Pv
#define SOA_I3		0.35	///< The allowable Current at 60V
#define TEMP_MAX	80	///< Maximum sink temperature before Eload switches off

typedef struct
{
	uint8_t	tempMax;
	double	pMax;	///< Up to this Voltage, 4A are allowable (20V for TIP102)
	double	v2;	///< Voltage Limit for Pv-Area (25V for TIP102)
	double	iMax;	///< The absolute maximum Current. Eload will not allow currents more than this.
	double	i2;	///< Current Limit for Pv-Area (3.5A for TIP102)
	double	i3;	///< Current Limit for 60V (350mA for TIP102)
} soa_t;

/*!	\brief Holds all Configuration Data stored in EEPROM

	All physical units are in SI base units or
	direct derivates
*/
typedef struct
{	uint16_t		magic;	//!< Used to discover unprogrammed Devices
	uint8_t	LCDV0;	//!< Control Voltage for LCD Contrast
	uint8_t	LCDBright;	//!< LCD Brightness
	double	P5VoltsPerLSB;	//!< Calibration Value for 5V
	double	P12VoltsPerLSB;	//!< Calibration Value for Unreg
	bool	backlightAlwaysOn;	///< True if the Backlight should stay on forever
	uint8_t	backlightOnTime;	///< Time the Backlight will stay on after Key Press in Seconds
	double	preset[10];	///< Value in Amperes for the 10 Presets
	double	ADCLowAmpsPerLSB;	///< Amperes per LSB in Low Current Range
	double	ADCHighAmpsPerLSB;	///< Amperes per LSB in High Current Range
	double	ADCDiffVoltsPerLSB;	///< Volts per LSB for differential Voltage measurement
	double	DACLowAmpsPerLSB;	///< Amperes per LSB in Low Current Range
	double	DACHighAmpsPerLSB;	///< Amperes per LSB in High Current Range
	soa_t	soa;	///< The Safe Operating Area
	#ifdef USE_RTS
	uint8_t	useRts;
	#endif
} confStruct_t;
extern bool	configDirty;

/*!	\brief These values will be calculated by ReadConfig()
	and/or on-the-fly by measurement

	All physical units are in SI base units or
	direct derivates
*/
typedef struct  
{	double	Rlow;		///< Value of the Sense Resistor for Low Current Range
	double	Rhigh;		///< Value of the Sense Resistor for High Current Range
	double	RchanP;		///< Calculated Channel Resistance of the FET shorting Rhigh || Rhigh
	double	dacHighAdj;	///< A factor considering RchanP
	double	rangeHighLimit;	///< Above this Current, Eload switches to high current Range
	double	rangeLowLimit;	///< Below this Current, Eload switches to low current Range
	double	VImax;		///< Maximum Voltage at IMax
	double	SBa;
	double	SBb;
} volatileConfStruct_t;

extern confStruct_t	config;
extern volatileConfStruct_t	volatileConfig;	///< Configuration Values that can be calculated
extern void	ReadConfig(void);
extern void	WriteConfig(void);
extern void	DumpConfigSignal(uint8_t number);
extern void	DumpConfig(void);
extern double SOAGetImax(double v);

#endif /* CONFIG_H_ */