﻿/*! \file config.c

	\brief	Routines for handling Configuration Data
	\author Robert Loos <robert.loos@loosweb.de>
	
	\copyright Copyright (C) 2014 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p>The routines here define the configuration memory and setting up the
default configuration.</p>
 */ 

#include <avr/eeprom.h>
#include <stdio.h>
#include <math.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include "eload.h"
#include "config.h"
#include "uart.h"
#include "delay.h"
#include "spi.h"

//! The Configuration. It is read on startup from the EEPROM.
confStruct_t	config;
volatileConfStruct_t	volatileConfig;	///< Configuration Values that can be calculated

/*!	\brief This struct will be stored in the EEPROM and is contained in
	the Production File.
	
	'New' devices are programmed using the production file which also can
	write the default EEPROM content and the fuses.
*/
confStruct_t	EEMEM eepConfig=
{
	.magic=CONF_MAGIC,	// Magic number to detect unprogrammed devices
	.LCDV0=45,	// Initial control voltage for the LC-Display
	.LCDBright=255,	// I love light! And I love power dissipation!
	.P5VoltsPerLSB=ADCVoltsPerLSB/dividerP5,	// Volts per LSB of the AD-Converter for 5V supply
	.P12VoltsPerLSB=ADCVoltsPerLSB/dividerP12,	// Volts per LSB of the AD-Converter for Unreg supply
	.backlightAlwaysOn=true,	// I mentioned before, I love light!
	.backlightOnTime=5,	// Backlight is switched off backlightOnTime seconds after the key has been pressed unless backlightAlwaysOn is true
	.preset= { PRESET0, PRESET1, PRESET2, PRESET3, PRESET4, PRESET5, PRESET6, PRESET7, PRESET8, PRESET9 },
	.ADCLowAmpsPerLSB=SPILOWAMPSPERLSB,
	.ADCHighAmpsPerLSB=SPIHIGHAMPSPERLSB,
	.ADCDiffVoltsPerLSB=SPIDIFFVOLTSPERLSB,
	.DACLowAmpsPerLSB=SPILOWAMPSPERLSB,
	.DACHighAmpsPerLSB=SPIHIGHAMPSPERLSB,
	.soa.iMax=SOA_IMAX,
	.soa.pMax=SOA_PMAX,
	.soa.i2=SOA_I2,
	.soa.v2=SOA_V2,
	.soa.i3=SOA_I3,
	.soa.tempMax=TEMP_MAX,
};

/*!	\brief	Loads default config values into conf-struct and stores it in non volatile Memory

	This function is mostly useful during development but I recommend to leave it functional.
	You can omit this routine if you want to save about 300 bytes of flash but it is
	quite useful if you change the config struct and change CONF_MAGIC. It will reprogram
	the EEPROM with default values then.
	Especially in devices already in the field it would lead to unpredictable results
	if the configuration memory format would not be compatible with the current version.
	\note Keep the values here in sync with the values in defaultConfig if you don't
	like surprises!
*/
static void DefaultConfig(void)
{
	delay1s();	// If you use the production file, EEPROM write may be interrupted by a reset if we don't a delay here!
	hprintf_P(PSTR("Writing Default Config.\n"));
	config.magic=CONF_MAGIC;	// Magic number to detect unprogrammed devices
	config.LCDV0=45;	// Initial control voltage for the LC-Display
	config.LCDBright=255;	// I love light! And I love power dissipation!
	config.P5VoltsPerLSB=ADCVoltsPerLSB/dividerP5;	// Volts per LSB of the AD-Converter for 5V supply
	config.P12VoltsPerLSB=ADCVoltsPerLSB/dividerP12;	// Volts per LSB of the AD-Converter for Unreg supply
	config.backlightAlwaysOn=true;	// I mentioned before, I love light!
	config.backlightOnTime=5;	// Backlight is switched off backlightOnTime seconds after the key has been pressed unless backlightAlwaysOn is true
	config.preset[0]= PRESET0;
	config.preset[1]= PRESET1;
	config.preset[2]= PRESET2;
	config.preset[3]= PRESET3;
	config.preset[4]= PRESET4;
	config.preset[5]= PRESET5;
	config.preset[6]= PRESET6;
	config.preset[7]= PRESET7;
	config.preset[8]= PRESET8;
	config.preset[9]= PRESET9;
	config.ADCLowAmpsPerLSB=SPILOWAMPSPERLSB;
	config.ADCHighAmpsPerLSB=SPIHIGHAMPSPERLSB;
	config.ADCDiffVoltsPerLSB=SPIDIFFVOLTSPERLSB;
	config.DACLowAmpsPerLSB=SPILOWAMPSPERLSB;
	config.DACHighAmpsPerLSB=SPIHIGHAMPSPERLSB;
	config.soa.iMax=SOA_IMAX,
	config.soa.pMax=SOA_PMAX,
	config.soa.i2=SOA_I2,
	config.soa.v2=SOA_V2,
	config.soa.i3=SOA_I3,
	config.soa.tempMax=TEMP_MAX,

	WriteConfig();
}

bool configDirty=false;	///< Set to true by routines affecting configuration if config memory needs to be saved to the EEPROM

/*!	\brief	Reads the Config-Struct from EEPROM into config.

	Only the <i>config</i>-struct in RAM will be used. Changes in <i>config</i> will only
	become permanent if you use a write-command. There is a <i>configDirty</i>-flag
	that is set by every routine that changes <i>config</i> and is used to remind the
	user to execute a write-command.
*/
void ReadConfig(void)
{
	eeprom_read_block(&config,&eepConfig,sizeof(confStruct_t));	// Copy the default config to the config hold in RAM
	hprintf_P(PSTR("Magic is %04x\n"),config.magic);
	if (config.magic!=CONF_MAGIC)	// no configuration data in EEPROM
	{
		DefaultConfig();	// Obsolete since we use EEMEM-Struct
	}
	configDirty=false;	// Configuration is valid from now
	volatileConfig.Rlow=SPIADCVOLTSPERLSB/config.ADCLowAmpsPerLSB;
	volatileConfig.Rhigh=SPIADCVOLTSPERLSB/config.ADCHighAmpsPerLSB;
	volatileConfig.RchanP=0.01f;	// Just a reasonable starting value. The exact value is calculated by measurement later.
	volatileConfig.dacHighAdj=1.0;	// Will be calculated later
	volatileConfig.rangeHighLimit=config.ADCLowAmpsPerLSB*65536.0*.95;
	volatileConfig.rangeLowLimit=volatileConfig.rangeHighLimit*.9;
	volatileConfig.VImax=config.soa.pMax/config.soa.iMax;
	volatileConfig.SBb=log(config.soa.i2/config.soa.i3)/log(config.soa.v2/60.0);
	volatileConfig.SBa=config.soa.i2/pow(config.soa.v2,volatileConfig.SBb);
	CalcIMaxLookupTable();
}

/*!	\brief	Writes the <i>Config</i>-Struct to EEPROM
*/
void WriteConfig(void)
{
	cli();	// Do not disturb!
	wdt_disable();
	eeprom_write_block(&config,&eepConfig,sizeof(confStruct_t));	// Copy the configuration data from RAM to EEPROM
	eeprom_busy_wait();
	configDirty=false;	// Configuration is valid now. RAM and EEPROM are the same
	wdt_reset();
	wdt_enable(WDTO_60MS);
	sei();
	hprintf_P(PSTR("Config written.\n"));
}

/*!	\brief	Calculate maximum allowable Current at a given Voltage
	\param	v The Voltage currently applied
	\return	The maximum allowable Current to stay within the SOA
*/
double SOAGetImax(double v)
{
	if (v<volatileConfig.VImax)	// Current is only limited by Imax
	{
		return config.soa.iMax;
	} else if (v<config.soa.v2)	// Constant PV area
	{
		return config.soa.pMax/v;
	} else
	{
		return volatileConfig.SBa*pow(v,volatileConfig.SBb);
	}
}

/*!	\brief	Lists the current configuration values

	Prints a verbose explanation of the configuration data
	to the host. This function relies on hprintf_P() which
	does the work and knows where the host is...
	\note	This function evaluates to nothing if SAVE_MEMORY is defined.
*/
void DumpConfig(void)
{
	hprintf_P(PSTR("\nCurrent configuration:\n"
		"LCD Brightness: %d\nLCD V0: %d\n"),config.LCDBright,config.LCDV0);
	hprintf_P(PSTR("Backlight always on: %d\nBacklight on time: %d\n"),config.backlightAlwaysOn,config.backlightOnTime);
	hprintf_P(PSTR("P5 mV per LSB: %.3f\nP12 mV per LSB: %.3f\n"),config.P5VoltsPerLSB*1000.0,config.P12VoltsPerLSB*1000.0);
	hprintf_P(PSTR("Load-Section:\nmV per LSB: %.3f\n"),config.ADCDiffVoltsPerLSB*1000.0);
	hprintf_P(PSTR("High Range uA per LSB: %.3f\n"),config.ADCHighAmpsPerLSB*1e6);
	hprintf_P(PSTR("Low Range uA per LSB: %.3f\n"),config.ADCLowAmpsPerLSB*1e6);
	hprintf_P(PSTR("DAC High Range uA per LSB: %.3f\n"),config.DACHighAmpsPerLSB*1e6);
	hprintf_P(PSTR("DAC Low Range uA per LSB: %.3f\n"),config.DACLowAmpsPerLSB*1e6);
	hprintf_P(PSTR("Presets:\n"));
	for (uint8_t i=0;i<=9;i++)
	{
		hprintf_P(PSTR("Preset %c: %.3fmA\n"),i+'0',config.preset[i]*1000.0);
	}
	hprintf_P(PSTR("SOA IMax: %5.3f PMax:%5.1f\nSOA I2: %5.3f V2: %5.2f\nSOA I3: %5.3f\nSBa %f SBb %f\n"),config.soa.iMax,config.soa.pMax,config.soa.i2,config.soa.v2,config.soa.i3,volatileConfig.SBa,volatileConfig.SBb);
	hprintf_P(PSTR("Volatile Config:\n"
		"rangeHighLimit: %1.3fA rangeLowLimit: %1.3fA\n"),volatileConfig.rangeHighLimit,volatileConfig.rangeLowLimit);
	hprintf_P(PSTR("Rlow: %fOhm\n"),volatileConfig.Rlow);
	hprintf_P(PSTR("Rhigh: %fOhm\n"),volatileConfig.Rhigh);
	hprintf_P(PSTR("Rchannel: %fOhm\n"),volatileConfig.RchanP);
	hprintf_P(PSTR("VImax: %5.2f\n"),volatileConfig.VImax);
}
