/*! \file eload.c

	\brief	The Main Program and miscellaneous Routines
	
	\copyright Copyright (C) 2009 Robert Loos	<http://www.loosweb.de>

	This program is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program.  If not, see <http://www.gnu.org/licenses/>.
<p></p>
 */ 

#include <avr/io.h>
#include <avr/lock.h>
#include <avr/wdt.h>
#include <avr/sleep.h>
#include <avr/pgmspace.h>
#include <avr/interrupt.h>
#include "adc.h"
#include "config.h"
#include "delay.h"
#include "Eload.h"
#include "keyboard.h"
#include "lcd.h"
#include "led.h"
#include "menu.h"
#include "screen.h"
#include "sensors.h"
#include "spi.h"
#include "uart.h"
#include "usartcmd.h"

const char *version="1.0";

/*!	\brief	Fuse settings for the Production ELF file
*/
FUSES =
{
	.low = (FUSE_SUT1 & FUSE_SUT0 & FUSE_CKSEL1),
	.high = (FUSE_SPIEN & FUSE_BOOTSZ1 & FUSE_BOOTSZ0),
	.extended = (FUSE_BODLEVEL0 & FUSE_BODLEVEL1),
};

LOCKBITS = (LB_MODE_1);	///< The Lock Bits for the production File

const char* build= __DATE__;	///< The Build Date as to be displayed in the Boot Screen

/*!	\brief	Flags set by Timer Interrupt to signal Main Loop

	The flags are set each second, minute, hour or day respectively by timer interrupt.
	The flags must be cleared by the main loop.
*/
volatile uint8_t	tFlags;		///< Flags set by Time
volatile uint8_t	dFlags;		///< Flags Controlling Debug Output
volatile uint8_t	eflags;		///< Error Flags
volatile uint8_t	tickCnt;	///< Counter for timer0 interrupts
volatile uint8_t	secCnt;		///< Seconds of the RTC
volatile uint8_t	minCnt;		///< Minutes of the RTC
volatile uint8_t	hrCnt;		///< Hours of the RTC
static volatile delay_t	delaycnt;	///< Counter for delay()
LED_FLASH_t	ledRedData;	///< This is the pattern the debug led should blink
LED_DATA_t	ledData;	///< This is the pattern the eight leds should blink. Every '1'-bit means the LED is on. Cycled in 200ms intervals from low to high
uint64_t	ahRaw;	///< The sum of current values
uint32_t	ahCnt;	///< Number of average values added to ahRaw;
uint32_t	ahTickCnt;	///< The number of 100Hz interrupts since ahRaw is counting

const char eflag1[] PROGMEM ="NTC 1 (case temp) malfunction\n";
const char eflag2[] PROGMEM ="NTC 2 (heat sink) malfunction\n";
const char eflag3[] PROGMEM ="+5V out of range\n";
const char eflag4[] PROGMEM ="+12V out of range\n";
const char eflag5[] PROGMEM ="SOA error\n";
const char eflag6[] PROGMEM ="Heat sink over temp\n";
const char eflag7[] PROGMEM ="Averages lost\n";
PGM_P const eflag_descr[] PROGMEM ={ eflag1,eflag2,eflag3,eflag4,eflag5,eflag6,eflag7 };

// Reset Counters in section .noinit because they must survive a reset!
uint8_t	wdrCnt __attribute__ ((section (".noinit")));	///< Counts the number of Watchdog Resets
uint8_t	boCnt __attribute__ ((section (".noinit")));	///< Counts the number of Brown Out Resets

void	Ticks2HMS(uint32_t ticks, char *str, size_t n)
{
	uint16_t	h=ticks/100/60/60;
	ticks-=(uint32_t)h*100*60*60;
	uint8_t		m=ticks/100/60;
	ticks-=(uint32_t)m*100*60;
	uint8_t		s=ticks/100;
	ticks-=s*100;
	uint8_t		t=ticks;
	snprintf_P(str,n,PSTR("%d:%02d:%02d.%02d"),h,m,s,t);
}

/*!	\brief	Timer0 compare interrupt (800Hz)

	The 800Hz-interrupt is not only used to create a RTC (Real Time Clock)
	but also for other tasks that rely on a constant timing, e.g.
	debouncing of the keyboard.<br>
	If you change the frequency, be sure of all of the consequences!
*/
ISR(TIMER0_COMPA_vect)
{
	static uint8_t	cnt;

sei();
	if (delaycnt)	// for delayms()
	{
		delaycnt--;
	}
	scanRotaryEncoder();
	cnt--;
	switch (cnt)
		{
			case 0:	// Every 10ms (100Hz)
				cnt=8;
				tickCnt++;
				ahTickCnt++;
				if (tickCnt==100) {	// One second has passed
					tickCnt=0;
					tFlags|=TFLAG_SEC;
					secCnt++;
					if (secCnt==60) {	// One minute has passed
						secCnt=0;	// Reset second counter
						tFlags|=TFLAG_MIN;
						minCnt++;
						if (minCnt==60) {	// One hour has passed
							minCnt=0;	//	Reset minute counter
							tFlags|=TFLAG_HOUR;
							hrCnt++;
							if (hrCnt==24) {	// One day has passed
								hrCnt=0;	// Reset hour counter
								tFlags|=TFLAG_DAY;	// And set day-flag. we don't care for longer time periods
							}
						}
					}
				}
				if ((tickCnt==0)||(tickCnt==50))	// Screen Updates two times a second
				{
					sFlags|=SFLAG_REFRESH;
				}
				break;
			case 1:
				UsartTimerCallback();	// Necessary for RTS/CTS-Handshake
				break;
			case 2:
				// scanKeyboard changes ports in a non-atomic way, so disable interrupts here!
				cli();
				scanKeyboard();
				sei();
				break;
			case 3:
				LedTimerCallback();
	}

}

/*!	\brief	Delays for n Milliseconds

	\note The timer resolution is 1.25ms and the state of the
		timer when calling this function is normally undefined.
		The actual delay may differ up to 1.25ms from the given
		value (especially, it may be 0 if you specify 1ms)!
	\param	ms The desired delay in Milliseconds
*/
void	DelayMs(uint8_t ms)
{
	delaycnt=((uint16_t)ms*8+4)/10;
	while (delaycnt!=0)
	{
		wdt_reset();
		sleep_cpu();
	}
}

/*!	\brief	Checks for Communication Flags

	\note Be sure to check and clear all Flags even if you don't
	use them to avoid unnecessary calls of this function
*/
static void DoCflags(void)
{
	if (cFlags&CFLAG_CHECKSUMERROR) {
		cFlags&=~CFLAG_CHECKSUMERROR;
	}
	if (cFlags&CFLAG_PARITYERROR) {
		cFlags&=~CFLAG_PARITYERROR;
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("%02d.%02d0 Parity Error\n"),secCnt,tickCnt);
		}
	}
	if (cFlags & CFLAG_CMDRECEIVED) { // lets look what the user wants...
		cFlags&=~CFLAG_CMDRECEIVED;
		ExecCommand((char*)cmdline);
		AssertRTS();
	}
	if (cFlags & CFLAG_BYTERECEIVED) // We are not interested in every single byte here
	{
		cFlags&=~CFLAG_BYTERECEIVED;
	}
}

/*!	\brief	Checks for Time Flags

	\note Be sure to check and clear all Flags even if you don't
	use them to avoid unnecessary calls of this function
*/
static void DoTflags(void)
{
	if (tFlags & TFLAG_SEC) {	// Do secondly stuff
		tFlags&=~TFLAG_SEC;	// Clear second-flag
#ifdef DEBUG
		if (dFlags&DFLAG_ADC)
		{
			hprintf_P(PSTR("SpiGetADC: rcv: %u %u %u range:%d avg: %u %u %u\n"),
				ADCGetIhighLifeRaw(),ADCGetIlowLifeRaw(),ADCGetVLifeRaw(),
				IsHighCurrentMode(),
				ADCGetIHighAvgRaw(),ADCGetILowAvgRaw(),ADCGetVAvgRaw());
		}
		if (dFlags&DFLAG_SENSOR)
		{
			hprintf_P(PSTR("P5:%.2f P12:%.2f DTemp:%.2f HSTemp:%.2f\n"),
				P5FromADC(ADCDataAvg.named.ADC_P5), P12FromADC(ADCDataAvg.named.ADC_P12),
				TempFromADC(),HSTempFromADC());
		}
		if (dFlags&DFLAG_AH)
		{
			char	str[20];
			
			Ticks2HMS(ahTickCnt,str,sizeof(str));
			hprintf_P(PSTR("ahRaw:%.0f ahCnt:%ld ahTickCnt:%ld "
				"ahRaw/ahCnt= %.2f Iavg:%.3f As:%.1f Time:%s\n"),
				(double)ahRaw,ahCnt,ahTickCnt,
				(double)ahRaw/ahCnt,ADCGetIByVal(ahRaw/ahCnt,true),ADCGetAs(),str);
		}
#endif
		ADCResetMinMax();
		CalcRChannel();
		for (uint8_t i=0;i<8;i++)
		{
//			hprintf_P(PSTR("ADC %d:%d "),i,ADCDataAvg.array[i]);
		}
//		hprintf_P(PSTR("\n"));
//		hprintf_P(PSTR("5V: %2.3f 12V: %2.3f Temp: %2.1f\n"),P5FromADC(ADCDataAvg.named.ADC_P5),P12FromADC(ADCDataAvg.named.ADC_P12),TempFromADC());
		HSTempFromADC();	// Ensure that temp is monitored
		switch (secCnt) {
			}
		if (tFlags & TFLAG_MIN)	// Do minutely stuff
		{
			tFlags&=~TFLAG_MIN;
			if (tFlags & TFLAG_HOUR) {	// do hourly stuff
				tFlags&=~TFLAG_HOUR;	// clear hour-flag
				if (tFlags & TFLAG_DAY) {	// do daily stuff
					tFlags&=~TFLAG_DAY;	// clear day-flag
				}
			}
			if (tFlags&TFLAG_DAY)	// We have no daily jobs until now
			{
				tFlags&=~TFLAG_DAY;	// Just reset the flag else this function would be called every 10ms!
			}
		}
		LCDSecondlyCallback();
	}			
}

/*!	\brief	Process Communication-, Time- and Key-Flags
*/
void	DoFlags()
{
	if (cFlags)	// Communication flags
	{
//		hprintf_P(PSTR("CF:%02x\n"),cFlags);
		DoCflags();
	}
	if (tFlags)	// Time flags
	{
//		hprintf_P(PSTR("TF:%02x\n"),tFlags);
		DoTflags();
	}
	wdt_reset();
	sleep_enable();
	sleep_cpu();	// sleep until something happens...
	sleep_disable();
}

uint8_t	mcusrMirror __attribute__ ((section (".noinit")));	///< Copy of the MCUSR taken shortly after reset
void	GetMCUSR(void) __attribute__ ((naked)) __attribute__ ((section(".init3")));
/*!	\brief	Get a copy of the MCUSR and reset it to 0

	This has to be done as soon as possible after a reset
	and so the function has been put into section .init3.
	Also, the watchdog timer is disabled since it may be
	running. See datasheet for details.
	\note	Making and keeping the copy in mcusr_mirror
	is not really necessary but the main program can determine
	the reset cause if you do.
*/
void	GetMCUSR(void)
{
	mcusrMirror=MCUSR;
	MCUSR=0;
	wdt_disable();
}

/*!	\brief	Initializes the System
*/
void	Init(void)
{
	DDRA=0b00000000;	// All lines input
	DDRB=0b10111111;	// All lines output except MISO
	DDRC=0b11111111;
	DDRD=0b11000110;
	InitKeyboard();
	InitUart();
	ReadConfig();
	SPIInit();
	PORTC=0;
	delay50ms();
	LDCInit();
	InitADC();
	TCCR0A=0b00000010;	// CTC-mode
	TCCR0B=0b00000100;	// Set timer 0 to CTC-mode, clock as Clk/256
	OCR0A=35;			// gives 100Hz*8
	TIMSK0|=0b00000010;	// enable timer0 Output Compare Match Interrupt
	TCCR2A=0b10100011;	// Non-inverting PWM, Fast PWM 8-bit
	TCCR2B=0b00000001;	// Direct System Clock
	OCR2A=config.LCDV0;	// set LCD-Voltage
	OCR2B=config.LCDBright;		// default LCD brightness
	SetHighCurrentMode();
	LEDRedOff();
	SetLowCurrentMode();
	wdt_enable(WDTO_1S);
	sei();
}

/*!	\brief	Performs a Light Show (visible LED test)
*/
void	LightShow(void)
{
	for (uint8_t i=0x01;i!=0;i<<=1)
	{
		SPISetLEDs(~i);
		delay1s();
	}
	SPISetLEDs(0x00);
	DelayMs(255);
	SPISetLEDs(0xff);
	delay1s();
}

/*!	\brief	The Main Program
*/
int main(void)
{
	Init();
	DACSetRaw(0);
//	LightShow();
	ledRedData=					0b00010111;
	ledData.named.powerGreen=	0b10111111;
	ledData.named.powerRed=		0b00001111;
	ledData.named.rangeGreen=	0b00110011;
	ledData.named.rangeBlue=	0b11111101;
	ledData.named.overTemp=		0b00000100;
	ledData.named.SOA=			0b10001000;
	ledData.named.fuse=			0b00010000;
	ledData.named.spare=		0b10000000;
	LCDClear();
/*	for (uint8_t i=0xa0;i!=0x0;i++)
	{
		LCDWriteData(i);
	}
	delay1s(); */
	hprintf_P(PSTR("Eload Rev %s Build %s\n"),version,build);
/*	hprintf_P(PSTR("I(10) %f\n"),SOAGetImax(10));
	hprintf_P(PSTR("I(24) %f\n"),SOAGetImax(24));
	hprintf_P(PSTR("I(30) %f\n"),SOAGetImax(30));
	hprintf_P(PSTR("I(40) %f\n"),SOAGetImax(40));
	hprintf_P(PSTR("I(60) %f\n"),SOAGetImax(60));
	hprintf_P(PSTR("I(100) %f\n"),SOAGetImax(100));
	hprintf_P(PSTR("R25=%f %d\nR-40=%f %d\nR80=%f %d\n R120=%f %d\n"),
		RNTCfromTemp(25.0),ADCfromTemp(25.0),
		RNTCfromTemp(-40.0),ADCfromTemp(-40.0),
		RNTCfromTemp(80.0),ADCfromTemp(80.0),
		RNTCfromTemp(120.0),ADCfromTemp(120.0)
		); */
	uint8_t c=Screen(&screenVersion);
	hprintf_P(PSTR("Screen returned %02x (%c)\n"),c,c);
/*	char str[20];
	ahTickCnt=12;
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("tick %f,%s\n"),(double)ahTickCnt,str);
	ahTickCnt*=100;
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("tick %f,%s\n"),(double)ahTickCnt,str);
	ahTickCnt*=60;
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("tick %f,%s\n"),(double)ahTickCnt,str);
	ahTickCnt*=60;
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("tick %f,%s\n"),(double)ahTickCnt,str);
	ahTickCnt*=24;
	Ticks2HMS(ahTickCnt,str,sizeof(str));
	hprintf_P(PSTR("tick %f,%s\n"),(double)ahTickCnt,str);*/
	ledData.named.powerGreen=	LED_BLINK_ON;
	ledData.named.powerRed=		LED_BLINK_OFF;
	ledData.named.rangeGreen=	LED_BLINK_OFF;
	ledData.named.rangeBlue=	LED_BLINK_OFF;
	ledData.named.overTemp=		LED_BLINK_OFF;
	ledData.named.SOA=			LED_BLINK_OFF;
	ledData.named.fuse=			LED_BLINK_OFF;
	ledData.named.spare=		LED_BLINK_OFF;
	SetLowCurrentMode();
 	eflags|=EFLAG_SOA;
   while(1)
    {
		wdt_reset();
		ledRedData=0b00001111;
		MenuMain();
		hprintf_P(PSTR("*** Main Menu Returned! ***\n"));
    }
}